<?php

namespace Modules\Academic\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreExamTypeRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return $this->user()->role === 'admin';
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'academic_year_id' => [
                'required',
                'integer',
                'exists:academic_years,id'
            ],
            'name' => [
                'required',
                'string',
                'max:255',
                'unique:exam_types,name'
            ],
            'description' => [
                'nullable',
                'string',
                'max:500'
            ],
            'weight' => [
                'required',
                'numeric',
                'min:0',
                'max:100'
            ],
            'is_active' => [
                'boolean'
            ],
        ];
    }

    /**
     * Get custom validation messages.
     */
    public function messages(): array
    {
        return [
            'name.required' => 'Please enter the exam type name.',
            'name.unique' => 'An exam type with this name already exists.',
            'max_marks.required' => 'Please enter the maximum marks.',
            'max_marks.min' => 'Maximum marks must be at least 1.',
            'max_marks.max' => 'Maximum marks cannot exceed 1000.',
            'pass_marks.required' => 'Please enter the pass marks.',
            'pass_marks.min' => 'Pass marks must be at least 1.',
            'pass_marks.lte' => 'Pass marks cannot exceed maximum marks.',
            'weightage.min' => 'Weightage cannot be negative.',
            'weightage.max' => 'Weightage cannot exceed 100%.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            // Validate pass marks percentage
            if ($this->max_marks && $this->pass_marks) {
                $passPercentage = ($this->pass_marks / $this->max_marks) * 100;

                if ($passPercentage < 20) {
                    $validator->errors()->add('pass_marks', 'Pass marks should be at least 20% of maximum marks.');
                }

                if ($passPercentage > 80) {
                    $validator->errors()->add('pass_marks', 'Pass marks should not exceed 80% of maximum marks.');
                }
            }

            // Validate weightage if provided
            if ($this->weightage !== null) {
                // Check if total weightage with existing exam types would exceed 100%
                $totalWeightage = \Modules\Academic\Models\ExamType::where('is_active', true)
                    ->sum('weightage');

                if (($totalWeightage + $this->weightage) > 100) {
                    $validator->errors()->add('weightage', 'Total weightage of all active exam types cannot exceed 100%.');
                }
            }

            // Validate reasonable marks range
            if ($this->max_marks) {
                if ($this->max_marks < 10) {
                    $validator->errors()->add('max_marks', 'Maximum marks should be at least 10 for meaningful assessment.');
                }
            }
        });
    }
}
