<?php

namespace Modules\Academic\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Database\Factories\AcademicYearFactory;

class AcademicYear extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'start_date',
        'end_date',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'start_date' => 'date',
            'end_date' => 'date',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return AcademicYearFactory::new();
    }

    /**
     * Get the classrooms for this academic year
     */
    public function classrooms(): HasMany
    {
        return $this->hasMany(Classroom::class);
    }

    /**
     * Get the exam types for this academic year
     */
    public function examTypes(): HasMany
    {
        return $this->hasMany(ExamType::class);
    }

    /**
     * Get the grading rules for this academic year
     */
    public function gradingRules(): HasMany
    {
        return $this->hasMany(\Modules\Configuration\Models\GradingRule::class);
    }

    /**
     * Get the division rules for this academic year
     */
    public function divisionRules(): HasMany
    {
        return $this->hasMany(\Modules\Configuration\Models\DivisionRule::class);
    }

    /**
     * Get the enrollments for this academic year
     */
    public function enrollments(): HasMany
    {
        return $this->hasMany(\Modules\Enrollment\Models\Enrollment::class);
    }

    /**
     * Get the results for this academic year
     */
    public function results(): HasMany
    {
        return $this->hasMany(\Modules\Results\Models\Result::class);
    }

    /**
     * Scope to get only active academic years
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get the current active academic year
     */
    public static function current()
    {
        return static::where('is_active', true)->first();
    }
}
