<?php

namespace Modules\Academic\Policies;

use App\Models\User;
use Modules\Academic\Models\Classroom;
use Illuminate\Auth\Access\HandlesAuthorization;

class ClassroomPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any classrooms.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher', 'student', 'guardian']);
    }

    /**
     * Determine whether the user can view the classroom.
     */
    public function view(User $user, Classroom $classroom): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view classrooms where they teach
                return $classroom->subjects()
                    ->where('teacher_id', $user->id)
                    ->exists();
            
            case 'student':
                // Students can view their enrolled classrooms
                return $user->student && 
                       $classroom->enrollments()
                           ->where('student_id', $user->student->id)
                           ->where('is_active', true)
                           ->exists();
            
            case 'guardian':
                // Guardians can view classrooms where their students are enrolled
                return $user->guardian && 
                       $classroom->enrollments()
                           ->whereHas('student.guardians', function ($query) use ($user) {
                               $query->where('guardian_id', $user->guardian->id);
                           })
                           ->where('is_active', true)
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can create classrooms.
     */
    public function create(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can update the classroom.
     */
    public function update(User $user, Classroom $classroom): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can delete the classroom.
     */
    public function delete(User $user, Classroom $classroom): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can manage classroom subjects.
     */
    public function manageSubjects(User $user, Classroom $classroom): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view classroom students.
     */
    public function viewStudents(User $user, Classroom $classroom): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view students in classrooms where they teach
                return $classroom->subjects()
                    ->where('teacher_id', $user->id)
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can manage classroom enrollments.
     */
    public function manageEnrollments(User $user, Classroom $classroom): bool
    {
        return $user->role === 'admin';
    }
}
