<?php

namespace Modules\Academic\Policies;

use App\Models\User;
use Modules\Academic\Models\Subject;
use Modules\Teachers\Models\Teacher;
use Illuminate\Auth\Access\HandlesAuthorization;

class SubjectPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any subjects.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can view the subject.
     */
    public function view(User $user, Subject $subject): bool
    {
        if ($user->role === 'admin') {
            return true;
        }

        if ($user->role === 'teacher') {
            return $subject->teacher_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can create subjects.
     */
    public function create(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can update the subject.
     */
    public function update(User $user, Subject $subject): bool
    {
        if ($user->role === 'admin') {
            return true;
        }

        if ($user->role === 'teacher') {
            // Teachers can only update their own subjects (limited fields)
            return $subject->teacher_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the subject.
     */
    public function delete(User $user, Subject $subject): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can assign teachers.
     */
    public function assignTeacher(User $user, Subject $subject): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can toggle status.
     */
    public function toggleStatus(User $user, Subject $subject): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view subject statistics.
     */
    public function viewStatistics(User $user, Subject $subject): bool
    {
        if ($user->role === 'admin') {
            return true;
        }

        if ($user->role === 'teacher') {
            return $subject->teacher_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can manage results for this subject.
     */
    public function manageResults(User $user, Subject $subject): bool
    {
        if ($user->role === 'admin') {
            return true;
        }

        if ($user->role === 'teacher') {
            return $subject->teacher_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can view enrolled students.
     */
    public function viewEnrolledStudents(User $user, Subject $subject): bool
    {
        if ($user->role === 'admin') {
            return true;
        }

        if ($user->role === 'teacher') {
            return $subject->teacher_id === $user->id;
        }

        return false;
    }
}
