<?php

namespace Modules\Enrollment\Policies;

use App\Models\User;
use Modules\Enrollment\Models\Enrollment;
use Illuminate\Auth\Access\HandlesAuthorization;

class EnrollmentPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any enrollments.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can view the enrollment.
     */
    public function view(User $user, Enrollment $enrollment): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view enrollments for classrooms they teach
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                // Check if teacher teaches any subjects in the enrollment's classroom
                return $enrollment->classroom->subjects()
                    ->where('teacher_id', $userTeacher->user_id)
                    ->exists();
            
            case 'student':
                // Students can view their own enrollments
                $userStudent = $user->student;
                if (!$userStudent) {
                    return false;
                }

                return $enrollment->student_id === $userStudent->id;
            
            case 'guardian':
                // Guardians can view enrollments of their students
                $userGuardian = $user->guardian;
                if (!$userGuardian) {
                    return false;
                }

                return $userGuardian->students()
                    ->where('students.id', $enrollment->student_id)
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can create enrollments.
     */
    public function create(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can update the enrollment.
     */
    public function update(User $user, Enrollment $enrollment): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can delete the enrollment.
     */
    public function delete(User $user, Enrollment $enrollment): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can transfer students.
     */
    public function transfer(User $user, Enrollment $enrollment): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can perform bulk operations.
     */
    public function bulkEnroll(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view enrollment statistics.
     */
    public function viewStatistics(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can export enrollment data.
     */
    public function export(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can import enrollment data.
     */
    public function import(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view available students.
     */
    public function viewAvailableStudents(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can manage classroom enrollments.
     */
    public function manageClassroomEnrollments(User $user, $classroomId): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can manage enrollments for classrooms they teach
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                $classroom = \Modules\Academic\Models\Classroom::find($classroomId);
                if (!$classroom) {
                    return false;
                }

                return $classroom->subjects()
                    ->where('teacher_id', $userTeacher->user_id)
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can view enrollment history.
     */
    public function viewHistory(User $user, $studentId): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view history for students they teach
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                $student = \Modules\Students\Models\Student::find($studentId);
                if (!$student) {
                    return false;
                }

                // Check if teacher teaches any of the student's current or past classrooms
                return $student->enrollments()
                    ->whereHas('classroom.subjects', function ($query) use ($userTeacher) {
                        $query->where('teacher_id', $userTeacher->user_id);
                    })
                    ->exists();
            
            case 'student':
                // Students can view their own enrollment history
                $userStudent = $user->student;
                if (!$userStudent) {
                    return false;
                }

                return $userStudent->id === (int)$studentId;
            
            case 'guardian':
                // Guardians can view enrollment history of their students
                $userGuardian = $user->guardian;
                if (!$userGuardian) {
                    return false;
                }

                return $userGuardian->students()
                    ->where('students.id', $studentId)
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can toggle enrollment status.
     */
    public function toggleStatus(User $user, Enrollment $enrollment): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view classroom capacity.
     */
    public function viewCapacity(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can manage enrollment dates.
     */
    public function manageDates(User $user, Enrollment $enrollment): bool
    {
        return $user->role === 'admin';
    }
}
