<?php

namespace Modules\Guardians\Policies;

use App\Models\User;
use Modules\Guardians\Models\Guardian;
use Illuminate\Auth\Access\HandlesAuthorization;

class GuardianPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any guardians.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher', 'guardian']);
    }

    /**
     * Determine whether the user can view the guardian.
     */
    public function view(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view guardians of their students
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                // Check if teacher teaches any of the guardian's students
                return $guardian->students()
                    ->whereHas('enrollments', function ($query) use ($userTeacher) {
                        $query->whereHas('classroom.subjects', function ($subjectQuery) use ($userTeacher) {
                            $subjectQuery->where('teacher_id', $userTeacher->user_id);
                        })->where('is_active', true);
                    })
                    ->exists();
            
            case 'guardian':
                // Guardians can view themselves
                return $guardian->user_id === $user->id;
            
            case 'student':
                // Students can view their guardians
                $userStudent = $user->student;
                if (!$userStudent) {
                    return false;
                }

                return $guardian->students()
                    ->where('students.id', $userStudent->id)
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can create guardians.
     */
    public function create(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can update the guardian.
     */
    public function update(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'guardian':
                // Guardians can update limited fields of their own profile
                return $guardian->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can delete the guardian.
     */
    public function delete(User $user, Guardian $guardian): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view guardian's students.
     */
    public function viewStudents(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view students they teach
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                return $guardian->students()
                    ->whereHas('enrollments', function ($query) use ($userTeacher) {
                        $query->whereHas('classroom.subjects', function ($subjectQuery) use ($userTeacher) {
                            $subjectQuery->where('teacher_id', $userTeacher->user_id);
                        })->where('is_active', true);
                    })
                    ->exists();
            
            case 'guardian':
                // Guardians can view their own students
                return $guardian->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can view student results.
     */
    public function viewStudentResults(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view results for subjects they teach
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                return $guardian->students()
                    ->whereHas('results', function ($query) use ($userTeacher) {
                        $query->whereHas('classroom.subjects', function ($subjectQuery) use ($userTeacher) {
                            $subjectQuery->where('teacher_id', $userTeacher->user_id);
                        });
                    })
                    ->exists();
            
            case 'guardian':
                // Guardians can view their students' results
                return $guardian->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can manage student assignments.
     */
    public function manageStudents(User $user, Guardian $guardian): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can export guardian data.
     */
    public function export(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can import guardian data.
     */
    public function import(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view guardian dashboard.
     */
    public function viewDashboard(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'guardian':
                // Guardians can view their own dashboard
                return $guardian->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can view guardian statistics.
     */
    public function viewStatistics(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'guardian':
                // Guardians can view their own statistics
                return $guardian->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can communicate with guardian.
     */
    public function communicate(User $user, Guardian $guardian): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can communicate with guardians of their students
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                return $guardian->students()
                    ->whereHas('enrollments', function ($query) use ($userTeacher) {
                        $query->whereHas('classroom.subjects', function ($subjectQuery) use ($userTeacher) {
                            $subjectQuery->where('teacher_id', $userTeacher->user_id);
                        })->where('is_active', true);
                    })
                    ->exists();
            
            default:
                return false;
        }
    }
}
