<?php

namespace Modules\Results\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class ImportResultsRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return $this->user()->hasRole('admin') || $this->user()->hasRole('teacher');
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'csv_file' => [
                'required',
                'file',
                'mimes:csv,txt',
                'max:5120' // 5MB max
            ],
            'academic_year_id' => [
                'required',
                'integer',
                'exists:academic_years,id'
            ],
            'exam_type_id' => [
                'required',
                'integer',
                'exists:exam_types,id'
            ],
            'classroom_id' => [
                'required',
                'integer',
                'exists:classrooms,id'
            ],
        ];
    }

    /**
     * Get custom validation messages.
     */
    public function messages(): array
    {
        return [
            'csv_file.required' => 'Please select a CSV file to upload.',
            'csv_file.file' => 'The uploaded file is not valid.',
            'csv_file.mimes' => 'The file must be a CSV file.',
            'csv_file.max' => 'The file size cannot exceed 5MB.',
            'academic_year_id.required' => 'Please select an academic year.',
            'academic_year_id.exists' => 'The selected academic year does not exist.',
            'exam_type_id.required' => 'Please select an exam type.',
            'exam_type_id.exists' => 'The selected exam type does not exist.',
            'classroom_id.required' => 'Please select a classroom.',
            'classroom_id.exists' => 'The selected classroom does not exist.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            // Check if exam type belongs to the academic year
            if ($this->exam_type_id && $this->academic_year_id) {
                $examTypeValid = \Modules\Academic\Models\ExamType::where('id', $this->exam_type_id)
                    ->where('academic_year_id', $this->academic_year_id)
                    ->where('is_active', true)
                    ->exists();

                if (!$examTypeValid) {
                    $validator->errors()->add('exam_type_id', 'The exam type does not belong to the selected academic year.');
                }
            }

            // For teachers, check if they can manage this classroom
            if ($this->user()->hasRole('teacher')) {
                $canManageClassroom = \Modules\Academic\Models\ClassroomSubject::where('classroom_id', $this->classroom_id)
                    ->where('teacher_id', $this->user()->id)
                    ->exists();

                if (!$canManageClassroom) {
                    $validator->errors()->add('classroom_id', 'You are not authorized to manage results for this classroom.');
                }
            }
        });
    }
}
