<?php

namespace Modules\Results\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Result extends Model
{
    use HasFactory;

    protected $fillable = [
        'student_id',
        'subject_id',
        'classroom_id',
        'academic_year_id',
        'exam_type_id',
        'marks',
        'grade',
        'points',
        'teacher_id',
        'entered_at',
    ];

    protected function casts(): array
    {
        return [
            'marks' => 'integer',
            'points' => 'integer',
            'entered_at' => 'datetime',
        ];
    }

    /**
     * Get the student for this result
     */
    public function student(): BelongsTo
    {
        return $this->belongsTo(\Modules\Students\Models\Student::class);
    }

    /**
     * Get the subject for this result
     */
    public function subject(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\Subject::class);
    }

    /**
     * Get the classroom for this result
     */
    public function classroom(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\Classroom::class);
    }

    /**
     * Get the academic year for this result
     */
    public function academicYear(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\AcademicYear::class);
    }

    /**
     * Get the exam type for this result
     */
    public function examType(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\ExamType::class);
    }

    /**
     * Get the teacher who entered this result
     */
    public function teacher(): BelongsTo
    {
        return $this->belongsTo(\App\Models\User::class, 'teacher_id');
    }

    /**
     * Scope to get results for a specific student
     */
    public function scopeForStudent($query, $studentId)
    {
        return $query->where('student_id', $studentId);
    }

    /**
     * Scope to get results for a specific academic year
     */
    public function scopeForAcademicYear($query, $academicYearId)
    {
        return $query->where('academic_year_id', $academicYearId);
    }

    /**
     * Scope to get results for a specific exam type
     */
    public function scopeForExamType($query, $examTypeId)
    {
        return $query->where('exam_type_id', $examTypeId);
    }

    /**
     * Scope to get results for a specific classroom
     */
    public function scopeForClassroom($query, $classroomId)
    {
        return $query->where('classroom_id', $classroomId);
    }

    /**
     * Scope to get results for a specific subject
     */
    public function scopeForSubject($query, $subjectId)
    {
        return $query->where('subject_id', $subjectId);
    }

    /**
     * Boot method to automatically calculate grade and points
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($result) {
            if ($result->marks !== null && $result->academic_year_id) {
                // Only auto-calculate grade and points if they are not manually provided
                if (empty($result->grade)) {
                    $result->grade = \Modules\Configuration\Models\GradingRule::getGradeForMarks(
                        $result->marks,
                        $result->academic_year_id
                    );
                }
                if (empty($result->points)) {
                    $result->points = \Modules\Configuration\Models\GradingRule::getPointsForMarks(
                        $result->marks,
                        $result->academic_year_id
                    );
                }
            }
        });
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return \Database\Factories\ResultFactory::new();
    }
}
