<?php

namespace Modules\Results\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class StudentResultSummary extends Model
{
    use HasFactory;

    protected $fillable = [
        'student_id',
        'academic_year_id',
        'exam_type_id',
        'classroom_id',
        'total_marks',
        'average_marks',
        'total_points',
        'division',
        'rank',
        'subjects_count',
        'computed_at',
    ];

    protected function casts(): array
    {
        return [
            'total_marks' => 'integer',
            'average_marks' => 'decimal:2',
            'total_points' => 'integer',
            'rank' => 'integer',
            'subjects_count' => 'integer',
            'computed_at' => 'datetime',
        ];
    }

    /**
     * Get the student for this summary
     */
    public function student(): BelongsTo
    {
        return $this->belongsTo(\Modules\Students\Models\Student::class);
    }

    /**
     * Get the academic year for this summary
     */
    public function academicYear(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\AcademicYear::class);
    }

    /**
     * Get the exam type for this summary
     */
    public function examType(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\ExamType::class);
    }

    /**
     * Get the classroom for this summary
     */
    public function classroom(): BelongsTo
    {
        return $this->belongsTo(\Modules\Academic\Models\Classroom::class);
    }

    /**
     * Scope to get summaries for a specific student
     */
    public function scopeForStudent($query, $studentId)
    {
        return $query->where('student_id', $studentId);
    }

    /**
     * Scope to get summaries for a specific academic year
     */
    public function scopeForAcademicYear($query, $academicYearId)
    {
        return $query->where('academic_year_id', $academicYearId);
    }

    /**
     * Scope to get summaries for a specific exam type
     */
    public function scopeForExamType($query, $examTypeId)
    {
        return $query->where('exam_type_id', $examTypeId);
    }

    /**
     * Scope to get summaries for a specific classroom
     */
    public function scopeForClassroom($query, $classroomId)
    {
        return $query->where('classroom_id', $classroomId);
    }
}
