<?php

namespace Modules\Results\Policies;

use App\Models\User;
use Modules\Results\Models\Result;
use Illuminate\Auth\Access\HandlesAuthorization;

class ResultPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any results.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher', 'student', 'guardian']);
    }

    /**
     * Determine whether the user can view the result.
     */
    public function view(User $user, Result $result): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view results for subjects they teach in classrooms they're assigned to
                return \Modules\Academic\Models\ClassroomSubject::where('classroom_id', $result->classroom_id)
                    ->where('subject_id', $result->subject_id)
                    ->where('teacher_id', $user->id)
                    ->exists();
            
            case 'student':
                // Students can only view their own results
                return $result->student->user_id === $user->id;
            
            case 'guardian':
                // Guardians can view results of their students
                return $user->guardian && 
                       $user->guardian->students()->where('students.id', $result->student_id)->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can create results.
     */
    public function create(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can update the result.
     */
    public function update(User $user, Result $result): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can update results for subjects they teach
                return \Modules\Academic\Models\ClassroomSubject::where('classroom_id', $result->classroom_id)
                    ->where('subject_id', $result->subject_id)
                    ->where('teacher_id', $user->id)
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can delete the result.
     */
    public function delete(User $user, Result $result): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can delete results they entered for subjects they teach
                return $result->teacher_id === $user->id &&
                       \Modules\Academic\Models\ClassroomSubject::where('classroom_id', $result->classroom_id)
                           ->where('subject_id', $result->subject_id)
                           ->where('teacher_id', $user->id)
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can import results.
     */
    public function import(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can export results.
     */
    public function export(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can view results for a specific classroom.
     */
    public function viewClassroomResults(User $user, int $classroomId): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view results for classrooms where they teach
                return \Modules\Academic\Models\ClassroomSubject::where('classroom_id', $classroomId)
                    ->where('teacher_id', $user->id)
                    ->exists();
            
            case 'student':
                // Students can view results for their enrolled classroom
                return $user->student && 
                       \Modules\Enrollment\Models\Enrollment::where('student_id', $user->student->id)
                           ->where('classroom_id', $classroomId)
                           ->where('is_active', true)
                           ->exists();
            
            case 'guardian':
                // Guardians can view results for classrooms where their students are enrolled
                return $user->guardian && 
                       $user->guardian->students()
                           ->whereHas('enrollments', function ($query) use ($classroomId) {
                               $query->where('classroom_id', $classroomId)
                                     ->where('is_active', true);
                           })
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can manage results for a specific classroom.
     */
    public function manageClassroomResults(User $user, int $classroomId): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can manage results for classrooms where they teach
                return \Modules\Academic\Models\ClassroomSubject::where('classroom_id', $classroomId)
                    ->where('teacher_id', $user->id)
                    ->exists();
            
            default:
                return false;
        }
    }
}
