<?php

namespace Modules\Students\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Student extends Model
{
    use HasFactory;

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return \Database\Factories\StudentFactory::new();
    }

    protected $fillable = [
        'user_id',
        'student_id',
        'first_name',
        'last_name',
        'date_of_birth',
        'gender',
        'address',
        'phone',
        'emergency_contact',
        'admission_date',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'date_of_birth' => 'date',
            'admission_date' => 'date',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get the user account for this student
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(\App\Models\User::class);
    }

    /**
     * Get the enrollments for this student
     */
    public function enrollments(): HasMany
    {
        return $this->hasMany(\Modules\Enrollment\Models\Enrollment::class);
    }

    /**
     * Get the classrooms this student is enrolled in
     */
    public function classrooms(): BelongsToMany
    {
        return $this->belongsToMany(
            \Modules\Academic\Models\Classroom::class,
            'enrollments',
            'student_id',
            'classroom_id'
        )->withPivot(['academic_year_id', 'enrolled_at'])
         ->withTimestamps();
    }

    /**
     * Get the guardians for this student
     */
    public function guardians(): BelongsToMany
    {
        return $this->belongsToMany(
            \Modules\Guardians\Models\Guardian::class,
            'student_guardians',
            'student_id',
            'guardian_id'
        )->withPivot(['relationship'])
         ->withTimestamps();
    }

    /**
     * Get the results for this student
     */
    public function results(): HasMany
    {
        return $this->hasMany(\Modules\Results\Models\Result::class);
    }

    /**
     * Get the full name of the student
     */
    public function getFullNameAttribute(): string
    {
        return "{$this->first_name} {$this->last_name}";
    }

    /**
     * Scope to get only active students
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get current enrollment for a specific academic year
     */
    public function getCurrentEnrollment($academicYearId = null)
    {
        $query = $this->enrollments();

        if ($academicYearId) {
            $query->where('academic_year_id', $academicYearId);
        } else {
            // Get current academic year enrollment
            $currentYear = \Modules\Academic\Models\AcademicYear::current();
            if ($currentYear) {
                $query->where('academic_year_id', $currentYear->id);
            }
        }

        return $query->first();
    }

    /**
     * Get results for a specific academic year and exam type
     */
    public function getResults($academicYearId, $examTypeId = null)
    {
        $query = $this->results()->where('academic_year_id', $academicYearId);

        if ($examTypeId) {
            $query->where('exam_type_id', $examTypeId);
        }

        return $query->with(['subject', 'examType'])->get();
    }
}
