<?php

namespace Modules\Students\Policies;

use App\Models\User;
use Modules\Students\Models\Student;
use Illuminate\Auth\Access\HandlesAuthorization;

class StudentPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any students.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher', 'student', 'guardian']);
    }

    /**
     * Determine whether the user can view the student.
     */
    public function view(User $user, Student $student): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view students in their classrooms
                return $student->enrollments()
                    ->whereHas('classroom.subjects', function ($query) use ($user) {
                        $query->where('teacher_id', $user->teacher->id);
                    })
                    ->where('is_active', true)
                    ->exists();
            
            case 'student':
                // Students can only view themselves
                return $student->user_id === $user->id;
            
            case 'guardian':
                // Guardians can view their students
                return $user->guardian && 
                       $student->guardians()
                           ->where('guardian_id', $user->guardian->id)
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can create students.
     */
    public function create(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can update the student.
     */
    public function update(User $user, Student $student): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can update students in their classrooms (limited fields)
                return $student->enrollments()
                    ->whereHas('classroom.subjects', function ($query) use ($user) {
                        $query->where('teacher_id', $user->teacher->id);
                    })
                    ->where('is_active', true)
                    ->exists();
            
            case 'student':
                // Students can update limited fields of their own profile
                return $student->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can delete the student.
     */
    public function delete(User $user, Student $student): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view student results.
     */
    public function viewResults(User $user, Student $student): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view results of students in their classrooms
                return $student->enrollments()
                    ->whereHas('classroom.subjects', function ($query) use ($user) {
                        $query->where('teacher_id', $user->teacher->id);
                    })
                    ->where('is_active', true)
                    ->exists();
            
            case 'student':
                // Students can view their own results
                return $student->user_id === $user->id;
            
            case 'guardian':
                // Guardians can view their students' results
                return $user->guardian && 
                       $student->guardians()
                           ->where('guardian_id', $user->guardian->id)
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can manage student guardians.
     */
    public function manageGuardians(User $user, Student $student): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can transfer the student.
     */
    public function transfer(User $user, Student $student): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view student enrollment history.
     */
    public function viewEnrollmentHistory(User $user, Student $student): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view enrollment history of students in their classrooms
                return $student->enrollments()
                    ->whereHas('classroom.subjects', function ($query) use ($user) {
                        $query->where('teacher_id', $user->teacher->id);
                    })
                    ->exists();
            
            case 'student':
                // Students can view their own enrollment history
                return $student->user_id === $user->id;
            
            case 'guardian':
                // Guardians can view their students' enrollment history
                return $user->guardian && 
                       $student->guardians()
                           ->where('guardian_id', $user->guardian->id)
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can view student performance summary.
     */
    public function viewPerformanceSummary(User $user, Student $student): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view performance summary of students in their classrooms
                return $student->enrollments()
                    ->whereHas('classroom.subjects', function ($query) use ($user) {
                        $query->where('teacher_id', $user->teacher->id);
                    })
                    ->where('is_active', true)
                    ->exists();
            
            case 'student':
                // Students can view their own performance summary
                return $student->user_id === $user->id;
            
            case 'guardian':
                // Guardians can view their students' performance summary
                return $user->guardian && 
                       $student->guardians()
                           ->where('guardian_id', $user->guardian->id)
                           ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can export student data.
     */
    public function export(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can import student data.
     */
    public function import(User $user): bool
    {
        return $user->role === 'admin';
    }
}
