<?php

namespace Modules\Teachers\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateTeacherRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return $this->user()->isAdmin();
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'teacher_id' => [
                'nullable',
                'string',
                'max:50',
                Rule::unique('teachers', 'teacher_id')->ignore($this->teacher->id)
            ],
            'first_name' => [
                'required',
                'string',
                'max:100',
                'regex:/^[a-zA-Z\s\.\-\']+$/'
            ],
            'last_name' => [
                'required',
                'string',
                'max:100',
                'regex:/^[a-zA-Z\s\.\-\']+$/'
            ],
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore($this->teacher->user_id)
            ],
            'password' => [
                'nullable',
                'string',
                'min:8',
                'confirmed'
            ],
            'date_of_birth' => [
                'required',
                'date',
                'before:today',
                'after:' . now()->subYears(65)->format('Y-m-d')
            ],
            'gender' => [
                'required',
                'string',
                'in:male,female,other'
            ],
            'address' => [
                'nullable',
                'string',
                'max:500'
            ],
            'phone' => [
                'nullable',
                'string',
                'max:20',
                'regex:/^[\+]?[0-9\s\-\(\)]+$/'
            ],
            'qualification' => [
                'nullable',
                'string',
                'max:255'
            ],
            'hire_date' => [
                'required',
                'date',
                'before_or_equal:today'
            ],
            'subjects' => [
                'nullable',
                'array'
            ],
            'subjects.*' => [
                'integer',
                'exists:subjects,id'
            ],
        ];
    }

    /**
     * Get custom validation messages.
     */
    public function messages(): array
    {
        return [
            'teacher_id.unique' => 'This teacher ID is already taken.',
            'first_name.required' => 'Please enter the teacher\'s first name.',
            'first_name.regex' => 'First name should only contain letters, spaces, periods, hyphens, and apostrophes.',
            'last_name.required' => 'Please enter the teacher\'s last name.',
            'last_name.regex' => 'Last name should only contain letters, spaces, periods, hyphens, and apostrophes.',
            'email.required' => 'Please enter an email address.',
            'email.unique' => 'This email address is already registered.',
            'date_of_birth.required' => 'Please enter the date of birth.',
            'date_of_birth.before' => 'Date of birth must be in the past.',
            'date_of_birth.after' => 'Teacher must be under 65 years old.',
            'gender.required' => 'Please select a gender.',
            'phone.regex' => 'Please enter a valid phone number.',
            'hire_date.required' => 'Please enter the hire date.',
            'hire_date.before_or_equal' => 'Hire date cannot be in the future.',
            'subjects.*.exists' => 'Selected subject does not exist.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            // Check if teacher is at least 18 years old
            if ($this->date_of_birth) {
                $birthDate = \Carbon\Carbon::parse($this->date_of_birth);
                $age = (int) $birthDate->diffInYears(now());
                if ($age < 18) {
                    $validator->errors()->add('date_of_birth', 'Teacher must be at least 18 years old.');
                }
            }

            // Validate hire date is not before birth date
            if ($this->date_of_birth && $this->hire_date) {
                $birthDate = \Carbon\Carbon::parse($this->date_of_birth);
                $hireDate = \Carbon\Carbon::parse($this->hire_date);

                if ($hireDate->lt($birthDate->addYears(16))) {
                    $validator->errors()->add('hire_date', 'Hire date must be at least 16 years after birth date.');
                }
            }

            // Check if subjects are active
            if ($this->subjects && is_array($this->subjects)) {
                $inactiveSubjects = \Modules\Academic\Models\Subject::whereIn('id', $this->subjects)
                    ->where('is_active', false)
                    ->exists();

                if ($inactiveSubjects) {
                    $validator->errors()->add('subjects', 'Cannot assign inactive subjects.');
                }
            }

            // Prevent changing critical information if teacher has active assignments
            $teacher = $this->route('teacher');
            if ($teacher) {
                // Check if teacher has active classroom assignments
                $hasActiveAssignments = \Illuminate\Support\Facades\DB::table('classroom_subjects')
                    ->join('classrooms', 'classroom_subjects.classroom_id', '=', 'classrooms.id')
                    ->join('academic_years', 'classrooms.academic_year_id', '=', 'academic_years.id')
                    ->where('classroom_subjects.teacher_id', $teacher->user_id)
                    ->where('academic_years.is_active', true)
                    ->exists();

                if ($hasActiveAssignments) {
                    // Check if hire date is being changed significantly
                    if ($this->hire_date) {
                        $originalHireDate = \Carbon\Carbon::parse($teacher->hire_date);
                        $newHireDate = \Carbon\Carbon::parse($this->hire_date);
                        if ($originalHireDate->diffInDays($newHireDate) > 7) {
                            $validator->errors()->add('hire_date', 'Cannot significantly change hire date for teachers with active assignments.');
                        }
                    }

                    // Check if date of birth is being changed significantly
                    if ($this->date_of_birth) {
                        $originalDob = \Carbon\Carbon::parse($teacher->date_of_birth);
                        $newDob = \Carbon\Carbon::parse($this->date_of_birth);
                        if ($originalDob->diffInDays($newDob) > 7) {
                            $validator->errors()->add('date_of_birth', 'Cannot significantly change date of birth for teachers with active assignments.');
                        }
                    }
                }
            }
        });
    }

    /**
     * Get custom attributes for validator errors.
     */
    public function attributes(): array
    {
        return [
            'teacher_id' => 'teacher ID',
            'first_name' => 'first name',
            'last_name' => 'last name',
            'date_of_birth' => 'date of birth',
            'hire_date' => 'hire date',
            'subjects.*' => 'subject',
        ];
    }
}
