<?php

namespace Modules\Teachers\Policies;

use App\Models\User;
use Modules\Teachers\Models\Teacher;
use Illuminate\Auth\Access\HandlesAuthorization;

class TeacherPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any teachers.
     */
    public function viewAny(User $user): bool
    {
        return in_array($user->role, ['admin', 'teacher']);
    }

    /**
     * Determine whether the user can view the teacher.
     */
    public function view(User $user, Teacher $teacher): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view themselves and colleagues in same subjects
                if ($teacher->user_id === $user->id) {
                    return true;
                }

                // Check if they share any subjects
                return $teacher->subjects()
                    ->whereIn('id', function ($query) use ($user) {
                        $query->select('subject_id')
                              ->from('teacher_subjects')
                              ->where('teacher_id', $user->id);
                    })
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can create teachers.
     */
    public function create(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can update the teacher.
     */
    public function update(User $user, Teacher $teacher): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can update limited fields of their own profile
                return $teacher->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can delete the teacher.
     */
    public function delete(User $user, Teacher $teacher): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can manage teacher subjects.
     */
    public function manageSubjects(User $user, Teacher $teacher): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view teacher's classrooms.
     */
    public function viewClassrooms(User $user, Teacher $teacher): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view their own classrooms and colleagues' in shared subjects
                if ($teacher->user_id === $user->id) {
                    return true;
                }

                // Check if they share any subjects
                return $teacher->subjects()
                    ->whereIn('id', function ($query) use ($user) {
                        $query->select('subject_id')
                              ->from('teacher_subjects')
                              ->where('teacher_id', $user->id);
                    })
                    ->exists();
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can view teacher's students.
     */
    public function viewStudents(User $user, Teacher $teacher): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view their own students and colleagues' in shared classrooms
                if ($teacher->user_id === $user->id) {
                    return true;
                }

                // Check if they teach in the same classrooms
                $userTeacher = $user->teacher;
                if (!$userTeacher) {
                    return false;
                }

                $sharedClassrooms = \Illuminate\Support\Facades\DB::table('classroom_subjects as cs1')
                    ->join('classroom_subjects as cs2', 'cs1.classroom_id', '=', 'cs2.classroom_id')
                    ->where('cs1.teacher_id', $userTeacher->user_id)
                    ->where('cs2.teacher_id', $teacher->user_id)
                    ->exists();

                return $sharedClassrooms;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can view teacher statistics.
     */
    public function viewStatistics(User $user, Teacher $teacher): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view their own statistics
                return $teacher->user_id === $user->id;
            
            default:
                return false;
        }
    }

    /**
     * Determine whether the user can export teacher data.
     */
    public function export(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can import teacher data.
     */
    public function import(User $user): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can assign classrooms to teacher.
     */
    public function assignClassrooms(User $user, Teacher $teacher): bool
    {
        return $user->role === 'admin';
    }

    /**
     * Determine whether the user can view teacher workload.
     */
    public function viewWorkload(User $user, Teacher $teacher): bool
    {
        switch ($user->role) {
            case 'admin':
                return true;
            
            case 'teacher':
                // Teachers can view their own workload
                return $teacher->user_id === $user->id;
            
            default:
                return false;
        }
    }
}
