<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use Modules\Academic\Models\ExamType;
use Illuminate\Support\Facades\Gate;

class TestExamTypeAuth extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:exam-type-auth';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test ExamType authorization';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Testing ExamType Authorization...');
        $this->info('=====================================');
        
        // Get admin user
        $adminUser = User::where('role', 'admin')->first();
        if (!$adminUser) {
            $this->error('No admin user found!');
            return 1;
        }
        
        $this->info("Testing with admin user: {$adminUser->email}");
        
        // Get a sample exam type
        $examType = ExamType::first();
        if (!$examType) {
            $this->info('No exam types found, creating one...');
            $examType = ExamType::factory()->create();
        }
        
        $this->info("Testing with exam type: {$examType->name}");
        
        // Test authorization
        $this->info('Testing authorization policies...');
        
        // Test viewAny
        $canViewAny = Gate::forUser($adminUser)->allows('viewAny', ExamType::class);
        $this->line("Can view any exam types: " . ($canViewAny ? 'YES' : 'NO'));
        
        // Test view
        $canView = Gate::forUser($adminUser)->allows('view', $examType);
        $this->line("Can view exam type: " . ($canView ? 'YES' : 'NO'));
        
        // Test create
        $canCreate = Gate::forUser($adminUser)->allows('create', ExamType::class);
        $this->line("Can create exam type: " . ($canCreate ? 'YES' : 'NO'));
        
        // Test update
        $canUpdate = Gate::forUser($adminUser)->allows('update', $examType);
        $this->line("Can update exam type: " . ($canUpdate ? 'YES' : 'NO'));
        
        // Test delete
        $canDelete = Gate::forUser($adminUser)->allows('delete', $examType);
        $this->line("Can delete exam type: " . ($canDelete ? 'YES' : 'NO'));
        
        $this->info('=====================================');
        
        // Test with student user
        $studentUser = User::where('role', 'student')->first();
        if ($studentUser) {
            $this->info("Testing with student user: {$studentUser->email}");
            
            $canViewAnyStudent = Gate::forUser($studentUser)->allows('viewAny', ExamType::class);
            $this->line("Student can view any exam types: " . ($canViewAnyStudent ? 'YES' : 'NO'));
            
            $canViewStudent = Gate::forUser($studentUser)->allows('view', $examType);
            $this->line("Student can view exam type: " . ($canViewStudent ? 'YES' : 'NO'));
            
            $canCreateStudent = Gate::forUser($studentUser)->allows('create', ExamType::class);
            $this->line("Student can create exam type: " . ($canCreateStudent ? 'YES' : 'NO'));
        }
        
        return 0;
    }
}
