<?php

namespace Database\Factories;

use Illuminate\Database\Eloquent\Factories\Factory;
use Modules\Students\Models\Student;
use App\Models\User;

class StudentFactory extends Factory
{
    protected $model = Student::class;

    public function definition(): array
    {
        $firstName = $this->faker->firstName();
        $lastName = $this->faker->lastName();

        return [
            'user_id' => User::factory()->student(),
            'student_id' => $this->generateStudentId(),
            'first_name' => $firstName,
            'last_name' => $lastName,
            'date_of_birth' => $this->faker->dateTimeBetween('-18 years', '-5 years')->format('Y-m-d'),
            'gender' => $this->faker->randomElement(['male', 'female', 'other']),
            'address' => $this->faker->optional()->address(),
            'phone' => $this->faker->optional()->phoneNumber(),
            'emergency_contact' => $this->faker->optional()->phoneNumber(),
            'admission_date' => $this->faker->dateTimeBetween('-3 years', 'now')->format('Y-m-d'),
            'is_active' => true,
        ];
    }

    /**
     * Generate a unique student ID
     */
    private function generateStudentId(): string
    {
        $year = date('Y');

        // For testing, use a random number to avoid conflicts
        if (app()->environment('testing')) {
            $randomNumber = $this->faker->unique()->numberBetween(1000, 9999);
            return "STU{$year}{$randomNumber}";
        }

        $lastStudent = Student::where('student_id', 'like', "STU{$year}%")
            ->orderBy('student_id', 'desc')
            ->first();

        if ($lastStudent) {
            $lastNumber = (int) substr($lastStudent->student_id, -3);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return "STU{$year}" . str_pad($newNumber, 3, '0', STR_PAD_LEFT);
    }

    /**
     * Create an active student
     */
    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => true,
        ]);
    }

    /**
     * Create an inactive student
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }

    /**
     * Create a male student
     */
    public function male(): static
    {
        return $this->state(fn (array $attributes) => [
            'gender' => 'male',
        ]);
    }

    /**
     * Create a female student
     */
    public function female(): static
    {
        return $this->state(fn (array $attributes) => [
            'gender' => 'female',
        ]);
    }

    /**
     * Create a student with specific user
     */
    public function forUser($userId): static
    {
        return $this->state(fn (array $attributes) => [
            'user_id' => $userId,
        ]);
    }

    /**
     * Create a student with specific admission date
     */
    public function admittedOn($date): static
    {
        return $this->state(fn (array $attributes) => [
            'admission_date' => $date,
        ]);
    }
}
