<?php

namespace Database\Factories;

use Illuminate\Database\Eloquent\Factories\Factory;
use Modules\Teachers\Models\Teacher;
use App\Models\User;

class TeacherFactory extends Factory
{
    protected $model = Teacher::class;

    public function definition(): array
    {
        $firstName = $this->faker->firstName();
        $lastName = $this->faker->lastName();

        return [
            'user_id' => User::factory()->teacher(),
            'teacher_id' => $this->generateTeacherId(),
            'first_name' => $firstName,
            'last_name' => $lastName,
            'date_of_birth' => $this->faker->dateTimeBetween('-65 years', '-22 years')->format('Y-m-d'),
            'gender' => $this->faker->randomElement(['male', 'female', 'other']),
            'address' => $this->faker->optional()->address(),
            'phone' => $this->faker->phoneNumber(),
            'qualification' => $this->faker->randomElement([
                'Bachelor of Education',
                'Master of Education',
                'Bachelor of Science',
                'Master of Science',
                'Bachelor of Arts',
                'Master of Arts',
                'PhD in Education',
                'Diploma in Teaching'
            ]),
            'hire_date' => $this->faker->dateTimeBetween('-10 years', 'now')->format('Y-m-d'),
            'is_active' => true,
        ];
    }

    /**
     * Generate a unique teacher ID
     */
    private function generateTeacherId(): string
    {
        $year = date('Y');

        // For testing, use a random number to avoid conflicts
        if (app()->environment('testing')) {
            $randomNumber = $this->faker->unique()->numberBetween(1000, 9999);
            return "TCH{$year}{$randomNumber}";
        }

        $lastTeacher = Teacher::where('teacher_id', 'like', "TCH{$year}%")
            ->orderBy('teacher_id', 'desc')
            ->first();

        if ($lastTeacher) {
            $lastNumber = (int) substr($lastTeacher->teacher_id, -3);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return "TCH{$year}" . str_pad($newNumber, 3, '0', STR_PAD_LEFT);
    }

    /**
     * Create an active teacher
     */
    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => true,
        ]);
    }

    /**
     * Create an inactive teacher
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }

    /**
     * Create a male teacher
     */
    public function male(): static
    {
        return $this->state(fn (array $attributes) => [
            'gender' => 'male',
        ]);
    }

    /**
     * Create a female teacher
     */
    public function female(): static
    {
        return $this->state(fn (array $attributes) => [
            'gender' => 'female',
        ]);
    }

    /**
     * Create a teacher with specific user
     */
    public function forUser($userId): static
    {
        return $this->state(fn (array $attributes) => [
            'user_id' => $userId,
        ]);
    }

    /**
     * Create a teacher with specific qualification
     */
    public function withQualification($qualification): static
    {
        return $this->state(fn (array $attributes) => [
            'qualification' => $qualification,
        ]);
    }

    /**
     * Create a senior teacher (hired more than 5 years ago)
     */
    public function senior(): static
    {
        return $this->state(fn (array $attributes) => [
            'hire_date' => $this->faker->dateTimeBetween('-15 years', '-5 years')->format('Y-m-d'),
        ]);
    }

    /**
     * Create a new teacher (hired within last 2 years)
     */
    public function newHire(): static
    {
        return $this->state(fn (array $attributes) => [
            'hire_date' => $this->faker->dateTimeBetween('-2 years', 'now')->format('Y-m-d'),
        ]);
    }
}
