<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Modules\Configuration\Models\GradingRule;
use Modules\Configuration\Models\DivisionRule;
use Modules\Results\Models\Result;
use Modules\Results\Models\StudentResultSummary;
use Modules\Academic\Models\AcademicYear;

class FreshGradingSystemSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('🔄 Clearing existing grading and division rules...');

        // Clear existing data
        GradingRule::truncate();
        DivisionRule::truncate();
        Result::truncate();
        StudentResultSummary::truncate();

        $this->command->info('✅ Existing data cleared');

        // Get all academic years
        $academicYears = AcademicYear::all();

        foreach ($academicYears as $academicYear) {
            $this->command->info("📚 Creating grading rules for Academic Year: {$academicYear->name}");
            $this->createGradingRules($academicYear->id);

            $this->command->info("📊 Creating division rules for Academic Year: {$academicYear->name}");
            $this->createDivisionRules($academicYear->id);
        }

        $this->command->info('🎉 Fresh grading system seeded successfully!');
    }

    /**
     * Create grading rules for an academic year
     */
    private function createGradingRules(int $academicYearId): void
    {
        $gradingRules = [
            ['grade' => 'A', 'min_marks' => 81, 'max_marks' => 100, 'points' => 1],
            ['grade' => 'B', 'min_marks' => 61, 'max_marks' => 80, 'points' => 2],
            ['grade' => 'C', 'min_marks' => 41, 'max_marks' => 60, 'points' => 3],
            ['grade' => 'D', 'min_marks' => 21, 'max_marks' => 40, 'points' => 4],
            ['grade' => 'F', 'min_marks' => 0, 'max_marks' => 20, 'points' => 5],
        ];

        foreach ($gradingRules as $rule) {
            GradingRule::create([
                'academic_year_id' => $academicYearId,
                'grade' => $rule['grade'],
                'min_marks' => $rule['min_marks'],
                'max_marks' => $rule['max_marks'],
                'points' => $rule['points'],
            ]);
        }
    }

    /**
     * Create division rules for an academic year
     */
    private function createDivisionRules(int $academicYearId): void
    {
        $divisionRules = [
            ['division' => 'INCOMPLETE', 'min_points' => 0, 'max_points' => 6],
            ['division' => 'I', 'min_points' => 7, 'max_points' => 17],
            ['division' => 'II', 'min_points' => 18, 'max_points' => 20],
            ['division' => 'III', 'min_points' => 21, 'max_points' => 24],
            ['division' => 'IV', 'min_points' => 25, 'max_points' => 31],
            ['division' => 'FAIL', 'min_points' => 32, 'max_points' => 35],
            ['division' => 'N/A', 'min_points' => 36, 'max_points' => 999], // High max for "36 and above"
        ];

        foreach ($divisionRules as $rule) {
            DivisionRule::create([
                'academic_year_id' => $academicYearId,
                'division' => $rule['division'],
                'min_points' => $rule['min_points'],
                'max_points' => $rule['max_points'],
            ]);
        }
    }
}
