<?php

/**
 * Debug Toast Errors
 * This script helps identify specific JavaScript errors in the toast system
 */

echo "🔍 TOAST ERROR DEBUGGER\n";
echo "=" . str_repeat("=", 60) . "\n\n";

// Check if the toast composable file exists and has correct syntax
echo "📁 CHECKING TOAST FILES...\n";
echo "-" . str_repeat("-", 40) . "\n";

$files = [
    'resources/js/composables/useToast.ts',
    'resources/js/components/ui/toast/Toast.vue',
    'resources/js/components/ui/toast/ToastContainer.vue',
    'resources/js/components/ui/toast/index.ts',
    'resources/js/components/AppShell.vue'
];

foreach ($files as $file) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        echo "✅ {$file} exists\n";
        
        // Check for common syntax issues
        if ($file === 'resources/js/composables/useToast.ts') {
            if (strpos($content, 'export const useToast') !== false || strpos($content, 'export { useToast }') !== false) {
                echo "   ✅ useToast is exported\n";
            } else {
                echo "   ❌ useToast export not found\n";
            }
        }
        
        if ($file === 'resources/js/components/ui/toast/index.ts') {
            if (strpos($content, 'useToast') !== false) {
                echo "   ✅ useToast re-exported in index\n";
            } else {
                echo "   ❌ useToast not re-exported in index\n";
            }
        }
    } else {
        echo "❌ {$file} not found\n";
    }
}

echo "\n";

// Check specific import paths
echo "🔗 CHECKING IMPORT PATHS...\n";
echo "-" . str_repeat("-", 40) . "\n";

$subjectsIndex = 'resources/js/pages/Academic/Subjects/Index.vue';
if (file_exists($subjectsIndex)) {
    $content = file_get_contents($subjectsIndex);
    
    // Check import statement
    if (preg_match("/import.*useToast.*from\s+['\"]([^'\"]+)['\"]/", $content, $matches)) {
        $importPath = $matches[1];
        echo "✅ useToast import found: {$importPath}\n";
        
        // Check if the import path resolves correctly
        $resolvedPath = str_replace('@/', 'resources/js/', $importPath);
        if (file_exists($resolvedPath . '.ts') || file_exists($resolvedPath . '.js')) {
            echo "   ✅ Import path resolves correctly\n";
        } else {
            echo "   ❌ Import path does not resolve: {$resolvedPath}\n";
        }
    } else {
        echo "❌ useToast import not found in Subjects Index\n";
    }
    
    // Check if toast is being used
    if (strpos($content, 'const toast = useToast()') !== false) {
        echo "✅ toast instance created\n";
    } else {
        echo "❌ toast instance not created\n";
    }
} else {
    echo "❌ Subjects Index file not found\n";
}

echo "\n";

// Check if there are any obvious syntax errors in key files
echo "🔧 CHECKING FOR SYNTAX ISSUES...\n";
echo "-" . str_repeat("-", 40) . "\n";

$toastComposable = 'resources/js/composables/useToast.ts';
if (file_exists($toastComposable)) {
    $content = file_get_contents($toastComposable);
    
    // Check for common issues
    $issues = [];
    
    if (strpos($content, 'export const useToast') === false && strpos($content, 'export { useToast }') === false) {
        $issues[] = 'Missing export statement';
    }
    
    if (strpos($content, 'const toasts = ref') === false) {
        $issues[] = 'Missing toasts ref';
    }
    
    if (strpos($content, 'addToast') === false) {
        $issues[] = 'Missing addToast function';
    }
    
    if (empty($issues)) {
        echo "✅ useToast composable looks correct\n";
    } else {
        echo "❌ useToast composable issues:\n";
        foreach ($issues as $issue) {
            echo "   - {$issue}\n";
        }
    }
}

echo "\n";

// Check ToastContainer
$toastContainer = 'resources/js/components/ui/toast/ToastContainer.vue';
if (file_exists($toastContainer)) {
    $content = file_get_contents($toastContainer);
    
    $issues = [];
    
    if (strpos($content, "import { useToast }") === false) {
        $issues[] = 'Missing useToast import';
    }
    
    if (strpos($content, "const { toasts, removeToast } = useToast()") === false) {
        $issues[] = 'Not using global toast state';
    }
    
    if (empty($issues)) {
        echo "✅ ToastContainer looks correct\n";
    } else {
        echo "❌ ToastContainer issues:\n";
        foreach ($issues as $issue) {
            echo "   - {$issue}\n";
        }
    }
}

echo "\n";

// Provide debugging steps
echo "🐛 DEBUGGING STEPS:\n";
echo "=" . str_repeat("=", 60) . "\n";

echo "1. **Open browser and visit**: http://localhost:8000/subjects\n";
echo "2. **Open Developer Tools**: Press F12\n";
echo "3. **Check Console tab** for JavaScript errors\n";
echo "4. **Look for specific errors** like:\n";
echo "   - 'useToast is not defined'\n";
echo "   - 'Cannot resolve module'\n";
echo "   - 'TypeError: Cannot read property'\n";
echo "   - Import/export errors\n\n";

echo "5. **Check Network tab** for failed resource loads\n";
echo "6. **Check if ToastContainer is in DOM**:\n";
echo "   - Right-click page → Inspect\n";
echo "   - Search for 'toast' in HTML\n";
echo "   - Look for elements with class 'fixed top-4 right-4'\n\n";

echo "7. **Test toast manually in console**:\n";
echo "   - Type: window.Vue\n";
echo "   - Check if Vue is available\n";
echo "   - Look for any global errors\n\n";

echo "🔧 **QUICK FIXES TO TRY**:\n";
echo "-" . str_repeat("-", 40) . "\n";
echo "1. Clear browser cache (Ctrl+Shift+R)\n";
echo "2. Restart Vite dev server: npm run dev\n";
echo "3. Check if all imports use correct paths\n";
echo "4. Verify TypeScript compilation\n\n";

echo "📋 **NEXT STEPS**:\n";
echo "1. Visit the browser and check console\n";
echo "2. Report the specific JavaScript error message\n";
echo "3. We can then fix the exact issue\n\n";

echo "🎯 **Most likely causes**:\n";
echo "- Import path resolution issue\n";
echo "- TypeScript compilation error\n";
echo "- Missing dependency\n";
echo "- Circular import\n\n";

echo "✅ **Debug script complete!**\n";
echo "Please check the browser console and report any errors you see.\n";
