import { ref } from 'vue'

interface ToastItem {
  id: string
  type: 'success' | 'error' | 'warning' | 'info'
  message: string
  duration?: number
  autoClose?: boolean
}

const toasts = ref<ToastItem[]>([])

export const useToast = () => {
  const addToast = (toast: Omit<ToastItem, 'id'>) => {
    const id = Date.now().toString()
    toasts.value.push({
      id,
      duration: 5000,
      autoClose: true,
      ...toast
    })
  }

  const removeToast = (id: string) => {
    const index = toasts.value.findIndex(toast => toast.id === id)
    if (index > -1) {
      toasts.value.splice(index, 1)
    }
  }

  const success = (message: string, options?: Partial<Omit<ToastItem, 'id' | 'type' | 'message'>>) => {
    addToast({ type: 'success', message, ...options })
  }

  const error = (message: string, options?: Partial<Omit<ToastItem, 'id' | 'type' | 'message'>>) => {
    addToast({ type: 'error', message, ...options })
  }

  const warning = (message: string, options?: Partial<Omit<ToastItem, 'id' | 'type' | 'message'>>) => {
    addToast({ type: 'warning', message, ...options })
  }

  const info = (message: string, options?: Partial<Omit<ToastItem, 'id' | 'type' | 'message'>>) => {
    addToast({ type: 'info', message, ...options })
  }

  return {
    toasts,
    addToast,
    removeToast,
    success,
    error,
    warning,
    info
  }
}
