<?php

/**
 * Test Edit and Delete Functionality
 * This script tests the edit and delete functionality for all modules
 */

echo "🧪 Testing Edit and Delete Functionality\n";
echo "=" . str_repeat("=", 50) . "\n\n";

// Test routes exist
echo "🛣️  Testing Edit Routes...\n";
$editRoutes = [
    'students.edit' => 'Students Edit',
    'teachers.edit' => 'Teachers Edit',
    'guardians.edit' => 'Guardians Edit',
    'academic-years.edit' => 'Academic Years Edit',
    'subjects.edit' => 'Subjects Edit',
    'classrooms.edit' => 'Classrooms Edit',
    'exam-types.edit' => 'Exam Types Edit',
    'results.edit' => 'Results Edit',
];

foreach ($editRoutes as $route => $name) {
    $output = shell_exec("php artisan route:list --name=$route 2>&1");
    if (strpos($output, 'GET') !== false) {
        echo "✅ $name route exists\n";
    } else {
        echo "❌ $name route missing\n";
    }
}
echo "\n";

// Test Vue Edit components exist
echo "🎨 Testing Edit Vue Components...\n";
$editComponents = [
    'resources/js/pages/Students/Edit.vue' => 'Students Edit',
    'resources/js/pages/Teachers/Edit.vue' => 'Teachers Edit',
    'resources/js/pages/Guardians/Edit.vue' => 'Guardians Edit',
    'resources/js/pages/Academic/AcademicYears/Edit.vue' => 'Academic Years Edit',
    'resources/js/pages/Academic/Subjects/Edit.vue' => 'Subjects Edit',
    'resources/js/pages/Academic/Classrooms/Edit.vue' => 'Classrooms Edit',
    'resources/js/pages/Academic/ExamTypes/Edit.vue' => 'Exam Types Edit',
    'resources/js/pages/Results/Edit.vue' => 'Results Edit',
];

foreach ($editComponents as $file => $name) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        if (strpos($content, '<template>') !== false && strpos($content, 'useForm') !== false) {
            echo "✅ $name component exists and has proper structure\n";
        } else {
            echo "⚠️  $name component exists but may have issues\n";
        }
    } else {
        echo "❌ $name component missing\n";
    }
}
echo "\n";

// Test Show components exist (needed for redirects after edit)
echo "👁️  Testing Show Vue Components...\n";
$showComponents = [
    'resources/js/pages/Students/Show.vue' => 'Students Show',
    'resources/js/pages/Teachers/Show.vue' => 'Teachers Show',
    'resources/js/pages/Guardians/Show.vue' => 'Guardians Show',
    'resources/js/pages/Academic/AcademicYears/Show.vue' => 'Academic Years Show',
    'resources/js/pages/Academic/Subjects/Show.vue' => 'Subjects Show',
    'resources/js/pages/Academic/Classrooms/Show.vue' => 'Classrooms Show',
    'resources/js/pages/Academic/ExamTypes/Show.vue' => 'Exam Types Show',
    'resources/js/pages/Results/Show.vue' => 'Results Show',
];

$missingShowPages = [];
foreach ($showComponents as $file => $name) {
    if (file_exists($file)) {
        echo "✅ $name component exists\n";
    } else {
        echo "❌ $name component missing\n";
        $missingShowPages[] = $name;
    }
}
echo "\n";

// Test validation request classes
echo "📝 Testing Validation Request Classes...\n";
$validationClasses = [
    'Modules/Students/app/Http/Requests/UpdateStudentRequest.php' => 'Update Student Request',
    'Modules/Teachers/app/Http/Requests/UpdateTeacherRequest.php' => 'Update Teacher Request',
    'Modules/Guardians/app/Http/Requests/UpdateGuardianRequest.php' => 'Update Guardian Request',
    'Modules/Academic/app/Http/Requests/UpdateAcademicYearRequest.php' => 'Update Academic Year Request',
    'Modules/Academic/app/Http/Requests/UpdateSubjectRequest.php' => 'Update Subject Request',
    'Modules/Academic/app/Http/Requests/UpdateClassroomRequest.php' => 'Update Classroom Request',
    'Modules/Academic/app/Http/Requests/UpdateExamTypeRequest.php' => 'Update Exam Type Request',
    'Modules/Results/app/Http/Requests/UpdateResultRequest.php' => 'Update Result Request',
];

$missingValidationClasses = [];
foreach ($validationClasses as $file => $name) {
    if (file_exists($file)) {
        echo "✅ $name exists\n";
    } else {
        echo "❌ $name missing\n";
        $missingValidationClasses[] = $name;
    }
}
echo "\n";

// Test controller methods
echo "🎮 Testing Controller Edit Methods...\n";
$controllerMethods = [
    'Modules/Students/app/Http/Controllers/StudentsController.php' => ['edit', 'update'],
    'Modules/Teachers/app/Http/Controllers/TeachersController.php' => ['edit', 'update'],
    'Modules/Guardians/app/Http/Controllers/GuardiansController.php' => ['edit', 'update'],
    'Modules/Academic/app/Http/Controllers/AcademicYearController.php' => ['edit', 'update'],
    'Modules/Academic/app/Http/Controllers/SubjectController.php' => ['edit', 'update'],
    'Modules/Academic/app/Http/Controllers/ClassroomController.php' => ['edit', 'update'],
    'Modules/Academic/app/Http/Controllers/ExamTypeController.php' => ['edit', 'update'],
    'Modules/Results/app/Http/Controllers/ResultsController.php' => ['edit', 'update'],
];

foreach ($controllerMethods as $file => $methods) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        $hasEdit = strpos($content, 'public function edit(') !== false;
        $hasUpdate = strpos($content, 'public function update(') !== false;
        
        $controllerName = basename($file, '.php');
        if ($hasEdit && $hasUpdate) {
            echo "✅ $controllerName has edit and update methods\n";
        } else {
            echo "❌ $controllerName missing methods - Edit: " . ($hasEdit ? 'Yes' : 'No') . ", Update: " . ($hasUpdate ? 'Yes' : 'No') . "\n";
        }
    } else {
        echo "❌ " . basename($file, '.php') . " controller missing\n";
    }
}
echo "\n";

// Test database for sample data to edit
echo "🗄️  Testing Sample Data for Editing...\n";
try {
    $pdo = new PDO('mysql:host=127.0.0.1;dbname=schoolapp', 'root', '');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $tables = [
        'students' => 'Students',
        'teachers' => 'Teachers',
        'guardians' => 'Guardians',
        'academic_years' => 'Academic Years',
        'subjects' => 'Subjects',
        'classrooms' => 'Classrooms',
        'exam_types' => 'Exam Types',
        'results' => 'Results',
    ];
    
    foreach ($tables as $table => $name) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM $table");
        $count = $stmt->fetchColumn();
        
        if ($count > 0) {
            echo "✅ $name: $count records available for editing\n";
        } else {
            echo "❌ $name: No records to edit\n";
        }
    }
} catch (PDOException $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "\n";
}
echo "\n";

// Summary
echo "📊 EDIT/DELETE FUNCTIONALITY TEST SUMMARY\n";
echo "=" . str_repeat("=", 50) . "\n";

$issues = [];
if (!empty($missingShowPages)) {
    $issues[] = "Missing Show pages: " . implode(', ', $missingShowPages);
}
if (!empty($missingValidationClasses)) {
    $issues[] = "Missing validation classes: " . implode(', ', $missingValidationClasses);
}

if (empty($issues)) {
    echo "🎉 All edit/delete functionality appears to be implemented!\n";
} else {
    echo "⚠️  Issues found:\n";
    foreach ($issues as $issue) {
        echo "   - $issue\n";
    }
}

echo "\n🚀 MANUAL TESTING STEPS:\n";
echo "1. Access http://localhost:8000/students\n";
echo "2. Click 'Edit' on any student record\n";
echo "3. Verify the edit form loads with existing data\n";
echo "4. Make changes and submit the form\n";
echo "5. Verify the changes are saved\n";
echo "6. Test delete functionality via dropdown menu\n";
echo "7. Repeat for all other modules\n";
echo "\n";

echo "🔍 COMMON ISSUES TO CHECK:\n";
echo "- Edit forms not loading (missing Show pages)\n";
echo "- Form validation errors\n";
echo "- Data not pre-populating in forms\n";
echo "- Redirect errors after form submission\n";
echo "- Delete confirmation dialogs not working\n";
echo "- Relationship data not loading properly\n";
echo "\n";

echo "✨ Edit/Delete functionality test completed!\n";
