<?php

namespace Tests\Browser;

use Illuminate\Foundation\Testing\DatabaseMigrations;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;
use App\Models\User;
use Modules\Students\Models\Student;
use Modules\Academic\Models\AcademicYear;
use Modules\Academic\Models\Subject;

class SchoolManagementUITest extends DuskTestCase
{
    use DatabaseMigrations;

    protected $admin;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create admin user
        $this->admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@test.com',
            'name' => 'Admin User'
        ]);

        // Create sample data
        AcademicYear::create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        Subject::create([
            'name' => 'Mathematics',
            'code' => 'MATH',
            'description' => 'Mathematics subject',
            'is_active' => true,
        ]);
    }

    /** @test */
    public function admin_can_login_and_access_dashboard()
    {
        $this->browse(function (Browser $browser) {
            $browser->visit('/login')
                    ->type('email', $this->admin->email)
                    ->type('password', 'password')
                    ->press('Log in')
                    ->assertPathIs('/dashboard')
                    ->assertSee('Dashboard');
        });
    }

    /** @test */
    public function admin_can_navigate_to_students_page()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Students')
                    ->assertPathIs('/students')
                    ->assertSee('Students')
                    ->assertSee('Add Student');
        });
    }

    /** @test */
    public function admin_can_access_create_student_form()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/students')
                    ->clickLink('Add Student')
                    ->assertPathIs('/students/create')
                    ->assertSee('Create Student')
                    ->assertSee('Student Information')
                    ->assertPresent('input[name="first_name"]')
                    ->assertPresent('input[name="last_name"]')
                    ->assertPresent('input[name="email"]');
        });
    }

    /** @test */
    public function admin_can_create_student_through_ui()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/students/create')
                    ->type('first_name', 'John')
                    ->type('last_name', 'Doe')
                    ->type('email', 'john.doe@test.com')
                    ->type('student_id', 'STU001')
                    ->type('date_of_birth', '2008-01-01')
                    ->select('gender', 'male')
                    ->type('address', '123 Test Street')
                    ->type('phone', '555-0001')
                    ->type('emergency_contact', '555-0002')
                    ->type('admission_date', '2024-01-01')
                    ->press('Create Student')
                    ->assertPathIs('/students')
                    ->assertSee('John Doe')
                    ->assertSee('STU001');
        });
    }

    /** @test */
    public function admin_can_navigate_to_teachers_page()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Teachers')
                    ->assertPathIs('/teachers')
                    ->assertSee('Teachers')
                    ->assertSee('Add Teacher');
        });
    }

    /** @test */
    public function admin_can_access_create_teacher_form()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/teachers')
                    ->clickLink('Add Teacher')
                    ->assertPathIs('/teachers/create')
                    ->assertSee('Create Teacher')
                    ->assertSee('Teacher Information')
                    ->assertPresent('input[name="first_name"]')
                    ->assertPresent('input[name="last_name"]')
                    ->assertPresent('input[name="email"]');
        });
    }

    /** @test */
    public function admin_can_navigate_to_guardians_page()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Guardians')
                    ->assertPathIs('/guardians')
                    ->assertSee('Guardians')
                    ->assertSee('Add Guardian');
        });
    }

    /** @test */
    public function admin_can_navigate_to_academic_sections()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Subjects')
                    ->assertPathIs('/subjects')
                    ->assertSee('Subjects')
                    ->assertSee('Add Subject')
                    ->visit('/dashboard')
                    ->clickLink('Classrooms')
                    ->assertPathIs('/classrooms')
                    ->assertSee('Classrooms')
                    ->assertSee('Add Classroom')
                    ->visit('/dashboard')
                    ->clickLink('Exam Types')
                    ->assertPathIs('/exam-types')
                    ->assertSee('Exam Types')
                    ->assertSee('Add Exam Type');
        });
    }

    /** @test */
    public function admin_can_navigate_to_enrollment_page()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Enrollment')
                    ->assertPathIs('/enrollment')
                    ->assertSee('Enrollments')
                    ->assertSee('New Enrollment');
        });
    }

    /** @test */
    public function admin_can_navigate_to_results_page()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Results')
                    ->assertPathIs('/results')
                    ->assertSee('Results')
                    ->assertSee('Add Result');
        });
    }

    /** @test */
    public function admin_can_navigate_to_configuration_pages()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->clickLink('Academic Years')
                    ->assertPathIs('/academic-years')
                    ->assertSee('Academic Years')
                    ->assertSee('Add Academic Year')
                    ->visit('/dashboard')
                    ->clickLink('Configuration')
                    ->assertPathIs('/configuration')
                    ->assertSee('Configuration')
                    ->assertSee('Grading Rules')
                    ->assertSee('Division Rules');
        });
    }

    /** @test */
    public function pagination_works_correctly_on_students_page()
    {
        // Create multiple students to test pagination
        $students = Student::factory()->count(25)->create();

        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/students')
                    ->assertSee('Showing')
                    ->assertSee('of')
                    ->assertSee('results');
            
            // Check if pagination controls are present when there are more than 20 records
            if ($browser->element('.pagination')) {
                $browser->assertPresent('.pagination');
            }
        });
    }

    /** @test */
    public function search_functionality_works_on_students_page()
    {
        // Create a student with specific name
        $student = Student::factory()->create([
            'first_name' => 'SearchTest',
            'last_name' => 'Student'
        ]);

        $this->browse(function (Browser $browser) use ($student) {
            $browser->loginAs($this->admin)
                    ->visit('/students')
                    ->type('input[placeholder*="Search"]', 'SearchTest')
                    ->pause(1000) // Wait for debounced search
                    ->assertSee('SearchTest Student');
        });
    }

    /** @test */
    public function filter_functionality_works_on_students_page()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/students')
                    ->assertPresent('select') // Status filter dropdown
                    ->assertSee('Clear Filters');
        });
    }

    /** @test */
    public function admin_can_access_student_details()
    {
        $student = Student::factory()->create([
            'first_name' => 'Detail',
            'last_name' => 'Test'
        ]);

        $this->browse(function (Browser $browser) use ($student) {
            $browser->loginAs($this->admin)
                    ->visit('/students')
                    ->clickLink('Detail Test')
                    ->assertPathIs("/students/{$student->id}")
                    ->assertSee('Detail Test')
                    ->assertSee('Student Information');
        });
    }

    /** @test */
    public function responsive_design_works_on_mobile()
    {
        $this->browse(function (Browser $browser) {
            $browser->resize(375, 667) // iPhone size
                    ->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->assertSee('Dashboard')
                    ->visit('/students')
                    ->assertSee('Students');
        });
    }

    /** @test */
    public function admin_can_logout()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->admin)
                    ->visit('/dashboard')
                    ->click('button[aria-label="User menu"]') // Assuming there's a user menu
                    ->clickLink('Logout')
                    ->assertPathIs('/login');
        });
    }
}
