<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use Modules\Students\Models\Student;
use Modules\Enrollment\Models\Enrollment;
use Modules\Academic\Models\Classroom;
use Modules\Academic\Models\AcademicYear;
use Illuminate\Foundation\Testing\RefreshDatabase;

class EnrollmentEditValidationTest extends TestCase
{
    use RefreshDatabase;

    private User $adminUser;
    private Student $student;
    private Enrollment $enrollment;
    private AcademicYear $activeAcademicYear;
    private AcademicYear $inactiveAcademicYear;

    protected function setUp(): void
    {
        parent::setUp();

        // Create admin user
        $this->adminUser = User::factory()->create([
            'role' => 'admin',
            'name' => 'Admin User',
            'email' => 'admin@test.com'
        ]);

        // Create student user
        $studentUser = User::factory()->create([
            'role' => 'student',
            'name' => 'John Doe',
            'email' => 'john.doe@test.com'
        ]);

        // Create student
        $this->student = Student::factory()->create([
            'user_id' => $studentUser->id,
            'first_name' => 'John',
            'last_name' => 'Doe',
            'student_id' => 'STU001'
        ]);

        // Create active academic year with current dates
        $this->activeAcademicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true
        ]);

        // Create inactive academic year with past dates
        $this->inactiveAcademicYear = AcademicYear::factory()->create([
            'name' => '2023-2024',
            'start_date' => '2023-09-01',
            'end_date' => '2024-06-30',
            'is_active' => false
        ]);

        // Create classroom in active academic year
        $activeClassroom = Classroom::factory()->create([
            'name' => 'Grade 10A',
            'academic_year_id' => $this->activeAcademicYear->id
        ]);

        // Create enrollment
        $this->enrollment = Enrollment::factory()->create([
            'student_id' => $this->student->id,
            'classroom_id' => $activeClassroom->id,
            'academic_year_id' => $this->activeAcademicYear->id,
            'is_active' => true
        ]);
    }

    public function test_can_edit_enrollment_with_active_academic_year_classroom()
    {
        // Create another classroom in the same active academic year
        $targetClassroom = Classroom::factory()->create([
            'name' => 'Grade 10B',
            'academic_year_id' => $this->activeAcademicYear->id
        ]);

        $response = $this->actingAs($this->adminUser)
            ->put("/enrollments/{$this->enrollment->id}", [
                'classroom_id' => $targetClassroom->id,
                'academic_year_id' => $this->activeAcademicYear->id,
                'enrolled_at' => '2024-10-01', // Within academic year range
                'is_active' => true
            ]);

        $response->assertRedirect("/enrollments/{$this->enrollment->id}");
        $response->assertSessionHas('success', 'Enrollment updated successfully.');
    }

    public function test_cannot_edit_enrollment_to_inactive_academic_year_classroom()
    {
        // Create classroom in inactive academic year
        $inactiveClassroom = Classroom::factory()->create([
            'name' => 'Grade 9A',
            'academic_year_id' => $this->inactiveAcademicYear->id
        ]);

        $response = $this->actingAs($this->adminUser)
            ->put("/enrollments/{$this->enrollment->id}", [
                'classroom_id' => $inactiveClassroom->id,
                'academic_year_id' => $this->inactiveAcademicYear->id,
                'enrolled_at' => '2023-10-01', // Within inactive academic year range
                'is_active' => true
            ]);

        $response->assertSessionHasErrors(['classroom_id' => 'Cannot transfer to classroom with inactive academic year.']);
    }

    public function test_can_edit_enrollment_without_changing_classroom()
    {
        // Test editing other fields without changing classroom
        $response = $this->actingAs($this->adminUser)
            ->put("/enrollments/{$this->enrollment->id}", [
                'classroom_id' => $this->enrollment->classroom_id, // Same classroom
                'academic_year_id' => $this->enrollment->academic_year_id,
                'enrolled_at' => '2024-11-01', // Change enrollment date within range
                'is_active' => true
            ]);

        $response->assertRedirect("/enrollments/{$this->enrollment->id}");
        $response->assertSessionHas('success', 'Enrollment updated successfully.');
    }
}
