<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use Modules\Academic\Models\ExamType;
use Modules\Academic\Models\AcademicYear;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ExamTypeDeleteTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that exam type can be deleted successfully
     */
    public function test_exam_type_can_be_deleted()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@delete.test',
        ]);

        // Create academic year
        $academicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create exam type
        $examType = ExamType::factory()->create([
            'name' => 'Test Exam to Delete',
            'description' => 'This exam type will be deleted',
            'academic_year_id' => $academicYear->id,
            'weight' => 25.00,
            'is_active' => true,
        ]);

        echo "Created exam type with ID: {$examType->id}\n";
        echo "Name: {$examType->name}\n";

        // Verify exam type exists
        $this->assertDatabaseHas('exam_types', [
            'id' => $examType->id,
            'name' => 'Test Exam to Delete',
        ]);

        // Test delete as admin
        $response = $this->actingAs($adminUser)
            ->delete("/exam-types/{$examType->id}");

        echo "Delete Response Status: " . $response->getStatusCode() . "\n";

        if ($response->getStatusCode() !== 302) {
            echo "Response Content: " . $response->getContent() . "\n";
        } else {
            echo "Redirect Location: " . $response->headers->get('Location') . "\n";
        }

        // Should redirect to index page
        $response->assertStatus(302);
        $response->assertRedirect(route('exam-types.index'));

        // Verify the exam type was deleted from database
        $this->assertDatabaseMissing('exam_types', [
            'id' => $examType->id,
        ]);

        echo "✓ Exam type deleted successfully\n";
    }

    /**
     * Test that non-admin cannot delete
     */
    public function test_non_admin_cannot_delete()
    {
        // Create student user
        $studentUser = User::factory()->create([
            'role' => 'student',
            'email' => 'student@delete.test',
        ]);

        // Create academic year and exam type
        $academicYear = AcademicYear::factory()->create();
        $examType = ExamType::factory()->create([
            'academic_year_id' => $academicYear->id,
        ]);

        // Try to delete as student
        $response = $this->actingAs($studentUser)
            ->delete("/exam-types/{$examType->id}");

        echo "Student Delete Response Status: " . $response->getStatusCode() . "\n";

        // Should be forbidden
        $response->assertStatus(403);

        // Verify exam type still exists
        $this->assertDatabaseHas('exam_types', [
            'id' => $examType->id,
        ]);

        echo "✓ Student correctly blocked from deleting\n";
    }

    /**
     * Test that exam type with results cannot be deleted
     */
    public function test_exam_type_with_results_cannot_be_deleted()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@results.test',
        ]);

        // Create academic year and exam type
        $academicYear = AcademicYear::factory()->create();
        $examType = ExamType::factory()->create([
            'academic_year_id' => $academicYear->id,
        ]);

        // Create a result for this exam type
        \Modules\Results\Models\Result::factory()->create([
            'exam_type_id' => $examType->id,
        ]);

        echo "Created exam type with results\n";

        // Try to delete exam type with results
        $response = $this->actingAs($adminUser)
            ->delete("/exam-types/{$examType->id}");

        echo "Delete with Results Response Status: " . $response->getStatusCode() . "\n";

        // Should redirect back with error
        $response->assertStatus(302);
        $response->assertSessionHasErrors();

        // Verify exam type still exists
        $this->assertDatabaseHas('exam_types', [
            'id' => $examType->id,
        ]);

        echo "✓ Exam type with results correctly protected from deletion\n";
    }
}
