<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use Modules\Academic\Models\ExamType;
use Modules\Academic\Models\AcademicYear;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ExamTypeFrontendTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that edit page loads without JavaScript errors
     */
    public function test_edit_page_loads_with_required_data()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@frontend.test',
        ]);

        // Create academic years
        $academicYear1 = AcademicYear::factory()->create([
            'name' => '2023-2024',
            'start_date' => '2023-09-01',
            'end_date' => '2024-06-30',
            'is_active' => false,
        ]);

        $academicYear2 = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create exam type
        $examType = ExamType::factory()->create([
            'name' => 'Frontend Test Exam',
            'description' => 'Test exam for frontend testing',
            'academic_year_id' => $academicYear1->id,
            'weight' => 40.00,
            'is_active' => true,
        ]);

        // Test edit page
        $response = $this->actingAs($adminUser)
            ->get("/exam-types/{$examType->id}/edit");

        echo "Edit Page Response Status: " . $response->getStatusCode() . "\n";

        $response->assertStatus(200);

        // Check that the page has the required Inertia props
        $response->assertInertia(function ($page) use ($examType, $academicYear1) {
            $page->component('Academic/ExamTypes/Edit')
                 ->has('examType')
                 ->has('academicYears')
                 ->where('examType.id', $examType->id)
                 ->where('examType.name', 'Frontend Test Exam')
                 ->where('examType.academic_year_id', $academicYear1->id)
                 ->where('examType.weight', '40.00')
                 ->where('examType.description', 'Test exam for frontend testing')
                 ->where('examType.is_active', true);
        });

        echo "✓ Edit page loads with correct exam type data\n";

        // Verify academic years are provided
        $response->assertInertia(function ($page) {
            $page->has('academicYears')
                 ->where('academicYears.0.name', '2023-2024')
                 ->where('academicYears.1.name', '2024-2025');
        });

        echo "✓ Edit page loads with academic years data (fixes the JavaScript error)\n";
    }

    /**
     * Test that create page loads with required data
     */
    public function test_create_page_loads_with_required_data()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@create.test',
        ]);

        // Create academic years
        $academicYear1 = AcademicYear::factory()->create([
            'name' => '2023-2024',
        ]);

        $academicYear2 = AcademicYear::factory()->create([
            'name' => '2024-2025',
        ]);

        // Test create page
        $response = $this->actingAs($adminUser)
            ->get('/exam-types/create');

        echo "Create Page Response Status: " . $response->getStatusCode() . "\n";

        $response->assertStatus(200);

        // Check that the page has the required Inertia props
        $response->assertInertia(function ($page) {
            $page->component('Academic/ExamTypes/Create')
                 ->has('academicYears');
        });

        echo "✓ Create page loads with correct component\n";

        // Verify academic years are provided
        $response->assertInertia(function ($page) {
            $page->has('academicYears')
                 ->where('academicYears.0.name', '2023-2024')
                 ->where('academicYears.1.name', '2024-2025');
        });

        echo "✓ Create page loads with academic years data\n";
    }

    /**
     * Test that index page loads correctly
     */
    public function test_index_page_loads()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@index.test',
        ]);

        // Create some exam types
        $academicYear = AcademicYear::factory()->create();

        ExamType::factory()->create([
            'name' => 'Midterm Exam',
            'academic_year_id' => $academicYear->id,
        ]);

        ExamType::factory()->create([
            'name' => 'Final Exam',
            'academic_year_id' => $academicYear->id,
        ]);

        // Test index page
        $response = $this->actingAs($adminUser)
            ->get('/exam-types');

        echo "Index Page Response Status: " . $response->getStatusCode() . "\n";

        $response->assertStatus(200);

        // Check that the page has the required Inertia props
        $response->assertInertia(function ($page) {
            $page->component('Academic/ExamTypes/Index')
                 ->has('examTypes');
        });

        echo "✓ Index page loads with exam types data\n";
    }
}
