<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use Modules\Academic\Models\ExamType;
use Modules\Academic\Models\AcademicYear;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ExamTypePageTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that admin can access exam types index page and see the content
     */
    public function test_admin_can_access_exam_types_page()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@page.test',
        ]);

        // Create academic year
        $academicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create exam type
        $examType = ExamType::factory()->create([
            'name' => 'Page Test Exam',
            'description' => 'Test exam for page testing',
            'academic_year_id' => $academicYear->id,
        ]);

        // Test as admin
        $response = $this->actingAs($adminUser)->get('/exam-types');

        echo "Response Status: " . $response->getStatusCode() . "\n";
        
        if ($response->getStatusCode() === 200) {
            echo "✓ Page loaded successfully\n";
            
            // Check if it's an Inertia response
            $headers = $response->headers->all();
            if (isset($headers['x-inertia'])) {
                echo "✓ Inertia response detected\n";
                
                // Get the Inertia props
                $content = $response->getContent();
                $data = json_decode($content, true);
                
                if (isset($data['props'])) {
                    echo "✓ Inertia props found\n";
                    
                    // Check if exam types data is present
                    if (isset($data['props']['examTypes'])) {
                        echo "✓ Exam types data found in props\n";
                        $examTypesData = $data['props']['examTypes'];
                        
                        if (isset($examTypesData['data']) && is_array($examTypesData['data'])) {
                            echo "✓ Exam types array found with " . count($examTypesData['data']) . " items\n";
                            
                            // Look for our test exam type
                            $found = false;
                            foreach ($examTypesData['data'] as $item) {
                                if (isset($item['name']) && $item['name'] === 'Page Test Exam') {
                                    echo "✓ Found our test exam type in the data\n";
                                    echo "  - ID: " . ($item['id'] ?? 'N/A') . "\n";
                                    echo "  - Name: " . ($item['name'] ?? 'N/A') . "\n";
                                    echo "  - Description: " . ($item['description'] ?? 'N/A') . "\n";
                                    echo "  - Is Active: " . ($item['is_active'] ? 'Yes' : 'No') . "\n";
                                    $found = true;
                                    break;
                                }
                            }
                            
                            if (!$found) {
                                echo "✗ Our test exam type not found in the data\n";
                                echo "Available exam types:\n";
                                foreach ($examTypesData['data'] as $item) {
                                    echo "  - " . ($item['name'] ?? 'Unknown') . "\n";
                                }
                            }
                        } else {
                            echo "✗ Exam types data is not an array or is missing\n";
                        }
                    } else {
                        echo "✗ No exam types data in props\n";
                        echo "Available props keys: " . implode(', ', array_keys($data['props'])) . "\n";
                    }
                } else {
                    echo "✗ No Inertia props found\n";
                }
            } else {
                echo "✗ Not an Inertia response\n";
                echo "Response content preview: " . substr($response->getContent(), 0, 500) . "\n";
            }
        } else {
            echo "✗ Page failed to load\n";
            echo "Response content: " . $response->getContent() . "\n";
        }

        $response->assertStatus(200);
    }

    /**
     * Test the actual HTML structure that would be rendered
     */
    public function test_exam_types_page_structure()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@structure.test',
        ]);

        // Create academic year
        $academicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create multiple exam types
        $examTypes = ExamType::factory()->count(3)->create([
            'academic_year_id' => $academicYear->id,
        ]);

        // Test as admin
        $response = $this->actingAs($adminUser)->get('/exam-types');

        $response->assertStatus(200);
        
        // Since this is an Inertia app, let's check the component and props
        $content = $response->getContent();
        $data = json_decode($content, true);
        
        $this->assertArrayHasKey('component', $data);
        $this->assertArrayHasKey('props', $data);
        
        echo "Component: " . $data['component'] . "\n";
        
        // Check if the component is the expected one
        $this->assertStringContains('ExamType', $data['component']);
        
        // Check if exam types are in the props
        $this->assertArrayHasKey('examTypes', $data['props']);
        
        $examTypesData = $data['props']['examTypes'];
        $this->assertArrayHasKey('data', $examTypesData);
        $this->assertIsArray($examTypesData['data']);
        $this->assertGreaterThanOrEqual(3, count($examTypesData['data']));
        
        echo "✓ Found " . count($examTypesData['data']) . " exam types in the response\n";
        
        // Check the structure of the first exam type
        $firstExamType = $examTypesData['data'][0];
        $this->assertArrayHasKey('id', $firstExamType);
        $this->assertArrayHasKey('name', $firstExamType);
        $this->assertArrayHasKey('description', $firstExamType);
        $this->assertArrayHasKey('is_active', $firstExamType);
        
        echo "✓ Exam type structure is correct\n";
    }
}
