<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Models\User;
use Modules\Academic\Models\ExamType;
use Modules\Academic\Models\AcademicYear;

class ExamTypeRouteTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test exam types index route accessibility
     */
    public function test_exam_types_index_route_with_admin()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@route.test',
        ]);

        // Create academic year
        $academicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create exam type
        $examType = ExamType::factory()->create([
            'name' => 'Test Exam',
            'academic_year_id' => $academicYear->id,
        ]);

        // Test as admin
        $response = $this->actingAs($adminUser)->get('/exam-types');

        echo "Response Status: " . $response->getStatusCode() . "\n";

        if ($response->getStatusCode() !== 200) {
            echo "Response Content: " . $response->getContent() . "\n";
        }

        $response->assertStatus(200);
    }

    /**
     * Test exam types index route with student (should be forbidden)
     */
    public function test_exam_types_index_route_with_student()
    {
        // Create student user
        $studentUser = User::factory()->create([
            'role' => 'student',
            'email' => 'student@route.test',
        ]);

        // Test as student
        $response = $this->actingAs($studentUser)->get('/exam-types');

        echo "Student Response Status: " . $response->getStatusCode() . "\n";

        // Should be forbidden (403) or redirect
        $this->assertContains($response->getStatusCode(), [403, 302]);
    }

    /**
     * Test exam types show route
     */
    public function test_exam_types_show_route()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@show.test',
        ]);

        // Create academic year
        $academicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create exam type
        $examType = ExamType::factory()->create([
            'name' => 'Test Show Exam',
            'academic_year_id' => $academicYear->id,
        ]);

        // Test show route
        $response = $this->actingAs($adminUser)->get("/exam-types/{$examType->id}");

        echo "Show Route Status: " . $response->getStatusCode() . "\n";
        echo "Show Route URL: /exam-types/{$examType->id}\n";

        if ($response->getStatusCode() !== 200) {
            echo "Show Response Content: " . substr($response->getContent(), 0, 500) . "...\n";
        }

        $response->assertStatus(200);
    }

    /**
     * Test exam types edit route
     */
    public function test_exam_types_edit_route()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@edit.test',
        ]);

        // Create academic year
        $academicYear = AcademicYear::factory()->create([
            'name' => '2024-2025',
            'start_date' => '2024-09-01',
            'end_date' => '2025-06-30',
            'is_active' => true,
        ]);

        // Create exam type
        $examType = ExamType::factory()->create([
            'name' => 'Test Edit Exam',
            'academic_year_id' => $academicYear->id,
        ]);

        // Test edit route
        $response = $this->actingAs($adminUser)->get("/exam-types/{$examType->id}/edit");

        echo "Edit Route Status: " . $response->getStatusCode() . "\n";
        echo "Edit Route URL: /exam-types/{$examType->id}/edit\n";

        if ($response->getStatusCode() !== 200) {
            echo "Edit Response Content: " . substr($response->getContent(), 0, 500) . "...\n";
        }

        $response->assertStatus(200);
    }
}
