<?php

namespace Tests\Feature\Frontend;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Modules\Academic\Models\Subject;
use Modules\Academic\Models\Classroom;
use Modules\Academic\Models\AcademicYear;
use Modules\Academic\Models\ExamType;
use Modules\Students\Models\Student;
use Modules\Teachers\Models\Teacher;
use Modules\Users\Models\User;
use Spatie\Permission\Models\Role;

class ToastNotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create admin role and user
        $adminRole = Role::create(['name' => 'admin']);
        $this->admin = User::factory()->create([
            'email' => 'admin@test.com',
            'password' => bcrypt('password')
        ]);
        $this->admin->assignRole($adminRole);
    }

    /** @test */
    public function it_can_access_all_index_pages_without_errors()
    {
        $this->actingAs($this->admin);

        $routes = [
            'subjects.index' => 'Subjects',
            'classrooms.index' => 'Classrooms', 
            'academic-years.index' => 'Academic Years',
            'exam-types.index' => 'Exam Types',
            'students.index' => 'Students',
            'teachers.index' => 'Teachers',
        ];

        foreach ($routes as $route => $title) {
            $response = $this->get(route($route));
            
            $response->assertStatus(200);
            $response->assertSee($title);
            
            // Check for common Vue.js errors in the response
            $response->assertDontSee('ReferenceError');
            $response->assertDontSee('TypeError');
            $response->assertDontSee('SyntaxError');
            
            echo "✅ {$title} index page loads successfully\n";
        }
    }

    /** @test */
    public function it_can_access_all_create_pages_without_errors()
    {
        $this->actingAs($this->admin);

        // Create required data
        $academicYear = AcademicYear::factory()->create();

        $routes = [
            'subjects.create' => 'Create Subject',
            'classrooms.create' => 'Create Classroom',
            'academic-years.create' => 'Create Academic Year',
            'exam-types.create' => 'Create Exam Type',
            'students.create' => 'Create Student',
            'teachers.create' => 'Create Teacher',
        ];

        foreach ($routes as $route => $title) {
            $response = $this->get(route($route));
            
            $response->assertStatus(200);
            
            // Check for common Vue.js errors
            $response->assertDontSee('ReferenceError');
            $response->assertDontSee('TypeError');
            $response->assertDontSee('SyntaxError');
            
            echo "✅ {$title} page loads successfully\n";
        }
    }

    /** @test */
    public function it_shows_success_toast_on_subject_creation()
    {
        $this->actingAs($this->admin);

        $response = $this->post(route('subjects.store'), [
            'name' => 'Test Subject',
            'code' => 'TEST001',
            'description' => 'Test Description',
            'is_active' => true
        ]);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Subject created successfully.');
        
        echo "✅ Subject creation shows success message\n";
    }

    /** @test */
    public function it_shows_error_toast_on_subject_validation_failure()
    {
        $this->actingAs($this->admin);

        $response = $this->post(route('subjects.store'), [
            'name' => '', // Required field missing
            'code' => '',
            'description' => '',
            'is_active' => true
        ]);

        $response->assertSessionHasErrors(['name', 'code']);
        
        echo "✅ Subject validation errors are properly handled\n";
    }

    /** @test */
    public function it_shows_success_toast_on_classroom_creation()
    {
        $this->actingAs($this->admin);
        
        $academicYear = AcademicYear::factory()->create();

        $response = $this->post(route('classrooms.store'), [
            'name' => 'Test Classroom',
            'academic_year_id' => $academicYear->id,
            'capacity' => 30,
            'description' => 'Test Description'
        ]);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Classroom created successfully.');
        
        echo "✅ Classroom creation shows success message\n";
    }

    /** @test */
    public function it_shows_error_toast_on_classroom_validation_failure()
    {
        $this->actingAs($this->admin);

        $response = $this->post(route('classrooms.store'), [
            'name' => '', // Required field missing
            'academic_year_id' => 999, // Non-existent ID
            'capacity' => 'invalid', // Invalid type
        ]);

        $response->assertSessionHasErrors(['name', 'academic_year_id', 'capacity']);
        
        echo "✅ Classroom validation errors are properly handled\n";
    }

    /** @test */
    public function it_shows_success_toast_on_subject_update()
    {
        $this->actingAs($this->admin);
        
        $subject = Subject::factory()->create();

        $response = $this->put(route('subjects.update', $subject), [
            'name' => 'Updated Subject',
            'code' => 'UPD001',
            'description' => 'Updated Description',
            'is_active' => true
        ]);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Subject updated successfully.');
        
        echo "✅ Subject update shows success message\n";
    }

    /** @test */
    public function it_shows_success_toast_on_subject_deletion()
    {
        $this->actingAs($this->admin);
        
        $subject = Subject::factory()->create();

        $response = $this->delete(route('subjects.destroy', $subject));

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Subject deleted successfully.');
        
        echo "✅ Subject deletion shows success message\n";
    }

    /** @test */
    public function it_shows_error_toast_on_subject_deletion_with_constraints()
    {
        $this->actingAs($this->admin);
        
        $subject = Subject::factory()->create();
        $academicYear = AcademicYear::factory()->create();
        $classroom = Classroom::factory()->create(['academic_year_id' => $academicYear->id]);
        
        // Attach subject to classroom to create constraint
        $classroom->subjects()->attach($subject->id);

        $response = $this->delete(route('subjects.destroy', $subject));

        $response->assertSessionHasErrors(['subject']);
        
        echo "✅ Subject deletion with constraints shows error message\n";
    }

    /** @test */
    public function it_shows_success_toast_on_classroom_deletion()
    {
        $this->actingAs($this->admin);
        
        $academicYear = AcademicYear::factory()->create();
        $classroom = Classroom::factory()->create(['academic_year_id' => $academicYear->id]);

        $response = $this->delete(route('classrooms.destroy', $classroom));

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Classroom deleted successfully.');
        
        echo "✅ Classroom deletion shows success message\n";
    }

    /** @test */
    public function it_validates_unique_classroom_names_within_academic_year()
    {
        $this->actingAs($this->admin);
        
        $academicYear = AcademicYear::factory()->create();
        
        // Create first classroom
        Classroom::factory()->create([
            'name' => 'Form 1A',
            'academic_year_id' => $academicYear->id
        ]);

        // Try to create another classroom with same name in same academic year
        $response = $this->post(route('classrooms.store'), [
            'name' => 'Form 1A',
            'academic_year_id' => $academicYear->id,
            'capacity' => 30
        ]);

        $response->assertSessionHasErrors(['name']);
        
        echo "✅ Classroom unique validation works correctly\n";
    }

    /** @test */
    public function it_allows_same_classroom_names_in_different_academic_years()
    {
        $this->actingAs($this->admin);
        
        $academicYear1 = AcademicYear::factory()->create(['name' => '2023-2024']);
        $academicYear2 = AcademicYear::factory()->create(['name' => '2024-2025']);
        
        // Create classroom in first academic year
        Classroom::factory()->create([
            'name' => 'Form 1A',
            'academic_year_id' => $academicYear1->id
        ]);

        // Create classroom with same name in different academic year
        $response = $this->post(route('classrooms.store'), [
            'name' => 'Form 1A',
            'academic_year_id' => $academicYear2->id,
            'capacity' => 30
        ]);

        $response->assertRedirect();
        $response->assertSessionHas('success');
        
        echo "✅ Classroom names can be same across different academic years\n";
    }

    /** @test */
    public function it_can_edit_classroom_without_name_conflict()
    {
        $this->actingAs($this->admin);
        
        $academicYear = AcademicYear::factory()->create();
        $classroom = Classroom::factory()->create([
            'name' => 'Form 1A',
            'academic_year_id' => $academicYear->id
        ]);

        // Edit classroom keeping the same name
        $response = $this->put(route('classrooms.update', $classroom), [
            'name' => 'Form 1A', // Same name
            'academic_year_id' => $academicYear->id,
            'capacity' => 35,
            'description' => 'Updated description'
        ]);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Classroom updated successfully.');
        
        echo "✅ Classroom can be edited without name conflict\n";
    }
}
