<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class InertiaTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that Inertia is working properly
     */
    public function test_inertia_request_returns_json()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@inertia.test',
        ]);

        // Make an Inertia request (with proper headers)
        $response = $this->actingAs($adminUser)
            ->withHeaders([
                'X-Inertia' => 'true',
                'X-Inertia-Version' => '1.0',
            ])
            ->get('/exam-types');

        echo "Response Status: " . $response->getStatusCode() . "\n";
        echo "Response Headers:\n";
        foreach ($response->headers->all() as $key => $values) {
            echo "  $key: " . implode(', ', $values) . "\n";
        }

        // Check if it's an Inertia response
        if ($response->headers->has('x-inertia')) {
            echo "✓ Inertia response detected\n";
            
            $content = $response->getContent();
            $data = json_decode($content, true);
            
            if ($data) {
                echo "✓ Valid JSON response\n";
                echo "Component: " . ($data['component'] ?? 'N/A') . "\n";
                echo "Props keys: " . implode(', ', array_keys($data['props'] ?? [])) . "\n";
            } else {
                echo "✗ Invalid JSON response\n";
                echo "Content: " . substr($content, 0, 500) . "\n";
            }
        } else {
            echo "✗ Not an Inertia response\n";
            echo "Content preview: " . substr($response->getContent(), 0, 500) . "\n";
        }

        $response->assertStatus(200);
    }

    /**
     * Test regular request without Inertia headers
     */
    public function test_regular_request_returns_html()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@regular.test',
        ]);

        // Make a regular request (without Inertia headers)
        $response = $this->actingAs($adminUser)->get('/exam-types');

        echo "Regular Request Response Status: " . $response->getStatusCode() . "\n";
        
        $content = $response->getContent();
        
        if (strpos($content, '<!DOCTYPE html>') === 0) {
            echo "✓ HTML response detected\n";
            
            // Check if it contains the Inertia app div
            if (strpos($content, 'id="app"') !== false) {
                echo "✓ Inertia app div found\n";
            } else {
                echo "✗ No Inertia app div found\n";
            }
            
            // Check if it contains Vue/Inertia scripts
            if (strpos($content, 'inertia') !== false || strpos($content, 'vue') !== false) {
                echo "✓ Inertia/Vue scripts found\n";
            } else {
                echo "✗ No Inertia/Vue scripts found\n";
            }
        } else {
            echo "✗ Not an HTML response\n";
            echo "Content preview: " . substr($content, 0, 500) . "\n";
        }

        $response->assertStatus(200);
    }

    /**
     * Test dashboard to see if Inertia works there
     */
    public function test_dashboard_inertia()
    {
        // Create admin user
        $adminUser = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@dashboard.test',
        ]);

        // Test dashboard with Inertia headers
        $response = $this->actingAs($adminUser)
            ->withHeaders([
                'X-Inertia' => 'true',
                'X-Inertia-Version' => '1.0',
            ])
            ->get('/dashboard');

        echo "Dashboard Response Status: " . $response->getStatusCode() . "\n";
        
        if ($response->headers->has('x-inertia')) {
            echo "✓ Dashboard Inertia response detected\n";
            
            $content = $response->getContent();
            $data = json_decode($content, true);
            
            if ($data) {
                echo "✓ Dashboard Valid JSON response\n";
                echo "Dashboard Component: " . ($data['component'] ?? 'N/A') . "\n";
            }
        } else {
            echo "✗ Dashboard Not an Inertia response\n";
        }

        $response->assertStatus(200);
    }
}
