// This file is part of HFS - Copyright 2021-2023, Massimo Melina <a@rejetto.com> - License https://www.gnu.org/licenses/gpl-3.0.txt

import * as http from 'http'
import { defineConfig, subMultipleConfigs } from './config'
import { app } from './index'
import * as https from 'https'
import { watchLoad } from './watchLoad'
import { networkInterfaces } from 'os';
import { getConnections, newConnection } from './connections'
import { TLSSocket } from 'node:tls'
import open from 'open'
import {
    CFG, debounceAsync, ipForUrl, makeNetMatcher, MINUTE, objSameKeys, onlyTruthy, prefix, runAt, wait, xlate
} from './misc'
import { PORT_DISABLED, ADMIN_URI, IS_WINDOWS } from './const'
import findProcess from 'find-process'
import { anyAccountCanLoginAdmin } from './adminApis'
import _ from 'lodash'
import { X509Certificate } from 'crypto'
import events from './events'
import { isIPv6 } from 'net'
import { defaultBaseUrl } from './nat'
import { storedMap } from './persistence'
import { argv } from './argv'
import { consoleHint } from './consoleLog'

interface ServerExtra { name: string, error?: string, busy?: Promise<string> }
let httpSrv: undefined | http.Server & ServerExtra
let httpsSrv: undefined | http.Server & ServerExtra

const openBrowserAtStart = defineConfig('open_browser_at_start', true)

export const baseUrl = defineConfig(CFG.base_url, '',
    x => /(?<=\/\/)[^\/]+/.exec(x)?.[0]) // compiled is host only

export async function getBaseUrlOrDefault() {
    return baseUrl.get() || await defaultBaseUrl.get()
}

export function getHttpsWorkingPort() {
    return httpsSrv?.listening && (httpsSrv.address() as any)?.port
}

const commonServerOptions: http.ServerOptions = { requestTimeout: 0 }
// these are properties that can be assigned to the server object
const commonServerAssign = { headersTimeout: 30_000, timeout: MINUTE } // 'headersTimeout' is not recognized by type lib, and 'timeout' is not effective when passed in parameters

const readyToListen = Promise.all([ storedMap.isOpening(), events.once('app') ])

const considerHttp = debounceAsync(async () => {
    await readyToListen
    void stopServer(httpSrv)
    httpSrv = Object.assign(http.createServer(commonServerOptions, app.callback()), { name: 'http' }, commonServerAssign)
    const host = listenInterface.get()
    const port = portCfg.get()
    if (port === PORT_DISABLED) return
    if (!await startServer(httpSrv, { port, host }))
        if (port !== 80)
            return consoleHint(`try specifying a different port, enter this command: config ${portCfg.key()} 1080`)
        else if (!await startServer(httpSrv, { port: 8080, host }))
            return
    httpSrv.on('connection', newConnection)
    printUrls(httpSrv.name)
    if (openBrowserAtStart.get() && !argv.updated)
        openAdmin()
})

export const portCfg = defineConfig('port', 80)
const listenInterface = defineConfig('listen_interface', '')
subMultipleConfigs(considerHttp, [portCfg, listenInterface])

export function openAdmin() {
    for (const srv of [httpSrv, httpsSrv]) {
        const a = srv?.address()
        if (!a || typeof a === 'string') continue
        const i = listenInterface.get()
        // open() will fail with ::1, don't know why, as my browser correctly opens the resulting url
        const hostname = i === '::1' || i in genericInterfaceNames ? 'localhost' : i
        const baseUrl = `${srv!.name}://${hostname}:${a.port}`
        open(baseUrl + ADMIN_URI, { wait: true}).catch(async e => {
            console.debug(String(e))
            console.warn("cannot launch browser on this machine >PLEASE< open your browser and reach one of these (you may need a different address)",
                ...Object.values(await getUrls()).flat().map(x => '\n - ' + x + ADMIN_URI))
            if (! anyAccountCanLoginAdmin())
                consoleHint(`you can enter this command: create-admin YOUR_PASSWORD`)
        })
        return true
    }
    console.log("openAdmin failed")
}

export function getCertObject() {
    const c = cert.compiled()
    if (!c) return
    const all = new X509Certificate(c)
    const some = _.pick(all, ['subject', 'issuer', 'validFrom', 'validTo'])
    const ret = objSameKeys(some, v => v?.includes('=') ? Object.fromEntries(v.split('\n').map(x => x.split('='))) : v)
    return Object.assign(ret, { altNames: all.subjectAltName?.replace(/DNS:/g, '').split(/, */) })
}

const considerHttps = debounceAsync(async () => {
    await readyToListen
    void stopServer(httpsSrv)
    defaultBaseUrl.proto = 'http'
    defaultBaseUrl.port = getCurrentPort(httpSrv) ?? 0
    let port = httpsPortCfg.get()
    try {
        const moreOptions = Object.assign({}, ...await events.emitAsync('httpsServerOptions') || [])  // emitAsync returns an array of objects
        httpsSrv = Object.assign(
            https.createServer(port === PORT_DISABLED ? {} : {
                ...commonServerOptions,
                key: privateKey.compiled(),
                cert: cert.compiled(),
                ...moreOptions,
            }, app.callback()),
            { name: 'https' },
            commonServerAssign
        )
        if (port >= 0) {
            const certObj = getCertObject()
            if (certObj) {
                const cn = certObj.subject?.CN
                if (cn)
                    console.log("certificate loaded for", certObj.altNames?.join(' + ') || cn)
                const now = new Date()
                const from = new Date(certObj.validFrom)
                const to = new Date(certObj.validTo)
                updateError() // error will change at from and to dates of the certificate
                const cancelTo = runAt(to.getTime(), updateError)
                const cancelFrom = runAt(from.getTime(), updateError)
                httpsSrv.on('close', () => {
                    cancelTo()
                    cancelFrom()
                })
                function updateError() {
                    if (!httpsSrv) return
                    httpsSrv.error = from > now ? "certificate not valid yet" : to < now ? "certificate expired" : undefined
                }
            }
            const namesForOutput: any = { cert: 'certificate', private_key: 'private key' }
            for (const x of httpsNeeds)
                if (!x.get())
                    return httpsSrv.error = "missing " + namesForOutput[x.key()]
                else if (!x.compiled())
                    return httpsSrv.error = "cannot read " + namesForOutput[x.key()]
        }
    }
    catch(e: any) {
        httpsSrv ||= Object.assign(https.createServer({}), { name: 'https' }) // a dummy container, in case creation failed because of certificate errors
        httpsSrv.error = "bad private key or certificate"
        console.error("failed to create https server: check your private key and certificate", e.message)
        return
    }
    httpsSrv.on('connection', newConnection) // this event is emitted as soon as the tcp layer is connected
    httpsSrv.on('secureConnection', (socket: TLSSocket) => { // emitted when the TLS layer is connected
        for (const c of getConnections()) // TLSSocket shares the same ip:port, so we can find its matching Connection
            if (socket.remoteAddress === c.socket.remoteAddress
            && socket.remotePort === c.socket.remotePort)
                return c.socket.emit('secure', socket) // let know Connection about the secure socket
    })
    port = await startServer(httpsSrv, { port, host: listenInterface.get() })
    if (!port) return
    printUrls(httpsSrv.name)
    events.emit('httpsReady')
    defaultBaseUrl.proto = 'https'
    defaultBaseUrl.port = getCurrentPort(httpsSrv) ?? 0
}, { wait: 200 }) // give time to have key and cert ready

export const cert = defineConfig('cert', '' as string, load)
export const privateKey = defineConfig('private_key', '' as string, load)
const httpsNeeds = [cert, privateKey]

function load(v: string, { object }: any) {
    object.watcher?.unwatch()
    if (!v || v.includes('\n'))
        return v
    // v is a path, we'll watch the file for changes
    object.watcher = watchLoad(v, x => object.setCompiled(x), { immediateFirst: true })
    return ''
}

export const httpsPortCfg = defineConfig('https_port', PORT_DISABLED)
subMultipleConfigs(considerHttps, [httpsPortCfg, listenInterface, ...httpsNeeds])

const genericInterfaceNames = {
    '0.0.0.0': "any IPv4",
    '::': "any IPv6",
    '': "any network",
}

function renderHost(host: string) {
    return xlate(host, genericInterfaceNames)
}

interface StartServer { port: number, host?:string }
export function startServer(srv: typeof httpSrv, { port, host }: StartServer) {
    return new Promise<number>(async resolve => {
        if (!srv) return resolve(0)
        try {
            if (port === PORT_DISABLED)
                return resolve(0)
            if (!host && !await testIpV4()) // !host means ipV4+6, and if v4 port alone is busy, we won't be notified of the failure, so we'll first test it on its own
                throw srv.error
            // from a few tests, this seems enough to support the expect-100 http/1.1 mechanism, at least with curl -T, not used by chrome|firefox anyway
            srv.on('checkContinue', (req, res) => srv.emit('request', req, res))
            port = await listen(host)
            if (port)
                console.log(srv.name, "serving on", renderHost(host || ''), ':', port)
            resolve(port)
        }
        catch(e) {
            srv.error = String(e)
            console.error(srv.name, `couldn't listen on port ${port}:`, srv.error)
            resolve(0)
        }
    })

    async function testIpV4() {
        const res = await listen('0.0.0.0', true)
        await new Promise(res => srv?.close(res)) // close, if any, and wait
        return res > 0
    }

    function listen(host?: string, silence=false) {
        return new Promise<number>(async (resolve, reject) => {
            srv?.on('error', onError).listen({ port, host }, () => {
                const ad = srv.address()
                if (!ad)
                    return reject('no address')
                if (typeof ad === 'string') {
                    srv.close()
                    return reject('type of socket not supported')
                }
                srv.removeListener('error', onError) // necessary in case someone calls stop/start many times
                events.emit('listening', { server: srv, port: ad.port })
                resolve(ad.port)
            })

             async function onError(e?: Error) {
                if (!srv) return
                srv.error = String(e)
                srv.busy = undefined
                const { code } = e as any
                if (code)
                    srv.busy = findProcess('port', port).then(
                        res => res?.map(x => prefix("Service", x.name === 'svchost.exe' && x.cmd.split(x.name)[1]?.trim()) || x.name).join(' + '),
                        () => '')
                if (code === 'EACCES' && port < 1024 && !srv.busy) // on Windows, when port is used by a service, we get EACCES
                    srv.error = `lacking permission on port ${port}, try with permission (${IS_WINDOWS ? 'administrator' : 'sudo'}) or port > 1024`
                if (code === 'EADDRINUSE' || srv.busy)
                    srv.error = `port ${port} busy: ${await srv.busy || "unknown process"}`
                if (!silence)
                    console.error(srv.name, srv.error)
                resolve(0)
            }
        })
    }
}

export function stopServer(srv?: http.Server) {
    return new Promise(resolve => {
        if (!srv?.listening)
            return resolve(null)
        const ad = srv.address()
        if (ad && typeof ad !== 'string')
            console.log("stopped port", ad.port)
        srv.close(err => {
            if (err && (err as any).code !== 'ERR_SERVER_NOT_RUNNING')
                console.debug("failed to stop server", String(err))
            resolve(err)
        })
    })
}

function getCurrentPort(srv: typeof httpSrv) {
    return (srv?.address() as any)?.port as number | undefined
}

export async function getServerStatus(includeSrv=true) {
    return {
        http: await serverStatus(httpSrv, portCfg.get()),
        https: await serverStatus(httpsSrv, httpsPortCfg.get()),
    }

    async function serverStatus(srv: typeof httpSrv, configuredPort: number) {
        const busy = await srv?.busy
        await wait(0) // simple trick to wait for also .error to be updated. If this trickery becomes necessary elsewhere, then we should make also error a Promise.
        return {
            ..._.pick(srv, ['listening', 'error']),
            busy,
            port: getCurrentPort(srv) || configuredPort,
            configuredPort,
            srv: includeSrv ? srv : undefined,
        }
    }}

const ignore = /^(lo|.*loopback.*|virtualbox.*|.*\(wsl\).*|llw\d|awdl\d|utun\d|anpi\d)$/i // avoid giving too much information

// AKA auto-ip https://en.wikipedia.org/wiki/Link-local_address
const isLinkLocal = makeNetMatcher('169.254.0.0/16|FE80::/10')

export async function getIps(external=true) {
    const only = { '0.0.0.0': 'IPv4', '::' : 'IPv6' }[listenInterface.get()] || ''
    const ips = onlyTruthy(Object.entries(networkInterfaces()).flatMap(([name, nets]) =>
        nets && !ignore.test(name) && nets.map(net => !net.internal && (!only || only === net.family) && net.address)
    ))
    const e = external && defaultBaseUrl.externalIp
    if (e && !ips.includes(e))
        ips.push(e)
    const noLinkLocal = ips.filter(x => !isLinkLocal(x))
    const ret = _.sortBy(noLinkLocal.length ? noLinkLocal : ips, [
        x => x !== defaultBaseUrl.localIp, // use the "nat" info to put best ip first
        isIPv6 // false=IPV4 comes first
    ])
    defaultBaseUrl.localIp ||= ret[0] || ''
    return ret
}

export async function getUrls() {
    const on = listenInterface.get()
    const ips = on === renderHost(on) ? [on] : await getIps()
    return Object.fromEntries(onlyTruthy([httpSrv, httpsSrv].map(srv => {
        if (!srv?.listening)
            return false
        const port = (srv?.address() as any)?.port
        const appendPort = port === (srv.name === 'https' ? 443 : 80) ? '' : ':' + port
        const urls = ips.map(ip => `${srv.name}://${ipForUrl(ip)}${appendPort}`)
        return urls.length && [srv.name, urls]
    })))
}

function printUrls(srvName: string) {
    getUrls().then(urls =>
        _.each(urls[srvName], url =>
            console.log('serving on', url)))
}
